package io.intino.amidas.connectors.microsoft.teams;

import com.azure.identity.ClientSecretCredentialBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonPrimitive;
import com.microsoft.bot.connector.authentication.MicrosoftAppCredentials;
import com.microsoft.graph.authentication.TokenCredentialAuthProvider;
import com.microsoft.graph.core.ClientException;
import com.microsoft.graph.models.AadUserConversationMember;
import com.microsoft.graph.models.Channel;
import com.microsoft.graph.models.ConversationMember;
import com.microsoft.graph.models.Group;
import com.microsoft.graph.requests.ConversationMemberCollectionPage;
import com.microsoft.graph.requests.ConversationMemberCollectionResponse;
import com.microsoft.graph.requests.GraphServiceClient;
import com.microsoft.graph.requests.GroupCollectionPage;
import io.intino.alexandria.logger.Logger;
import io.intino.amidas.connectors.microsoft.teams.model.Member;
import io.intino.amidas.connectors.microsoft.teams.model.Team;
import okhttp3.Request;
import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;

public class TeamsInfo {
	private final MicrosoftAppCredentials credentials;

	public TeamsInfo(MicrosoftAppCredentials credentials) {
		this.credentials = credentials;
	}

	public List<Team> reloadTeams() {
		final GraphServiceClient<Request> graphClient = client();
		GroupCollectionPage response = graphClient.groups().buildRequest().get();
		return map(response.getCurrentPage(), graphClient);
	}

	public void createTeam(String name, List<String> users) {
		com.microsoft.graph.models.Team team = new com.microsoft.graph.models.Team();
		team.additionalDataManager().put("template@odata.bind", new JsonPrimitive("https://graph.microsoft.com/v1.0/teamsTemplates('standard')"));
		team.displayName = name;
		team.description = name;
		AadUserConversationMember members = new AadUserConversationMember();
		members.roles = List.of("owner");
		members.additionalDataManager().put("user@odata.bind", new JsonPrimitive("https://graph.microsoft.com/v1.0/users('0040b377-61d8-43db-94f5-81374122dc7e')"));
		ConversationMemberCollectionResponse conversationMemberCollectionResponse = new ConversationMemberCollectionResponse();
		conversationMemberCollectionResponse.value = List.of(members);
		team.members = new ConversationMemberCollectionPage(conversationMemberCollectionResponse, null);
		client().teams().buildRequest().post(team);
		addMembers(users);
	}

	private void addMembers(List<String> users) {
		LinkedList<ConversationMember> valuesList = new LinkedList<>();
		for (String user : users) {
			AadUserConversationMember values = new AadUserConversationMember();
			values.roles = List.of(user);
			values.additionalDataManager().put("user@odata.bind", new JsonPrimitive("https://graph.microsoft.com/v1.0/users('18a80140-b0fb-4489-b360-2f6efaf225a0')"));
			valuesList.add(values);
		}
		ConversationMemberCollectionResponse conversationMemberCollectionResponse = new ConversationMemberCollectionResponse();
		conversationMemberCollectionResponse.value = valuesList;
		ConversationMemberCollectionPage conversationMemberCollectionPage = new ConversationMemberCollectionPage(conversationMemberCollectionResponse, null);

	}

	private List<Member> extractMembers(GraphServiceClient<Request> graphClient, List<Team> teams) {
		List<Member> members = new ArrayList<>();
		for (Team team : teams) {
			members.addAll(map(graphClient.teams(team.id()).members().buildRequest().get().getCurrentPage()));
		}
		return members;
	}

	private List<Member> map(List<ConversationMember> page) {
		return page.stream().map(m -> new Member(m.id, m.displayName)).collect(Collectors.toList());
	}

	private List<Team> map(List<Group> groups, GraphServiceClient<Request> client) {
		List<Team> teams = new ArrayList<>();
		for (Group group : groups) {
			if (!isTeamsGroup(group)) continue;
			List<Channel> channels = channels(client, group);
			teams.add(new Team(group.id, group.displayName, channels.stream().map(c -> new Team.Channel(c.id, c.displayName, group.preferredLanguage)).collect(Collectors.toList())));
		}
		return teams;
	}

	private static boolean isTeamsGroup(Group group) {
		JsonElement element = group.additionalDataManager().get("resourceProvisioningOptions");
		if (element == null) return false;
		for (JsonElement e : element.getAsJsonArray()) if (e.getAsString().equals("Team")) return true;
		return false;
	}

	@NotNull
	private static List<Channel> channels(GraphServiceClient<Request> client, Group group) {
		try {
			return client.teams(group.id).channels().buildRequest().get().getCurrentPage();
		} catch (ClientException e) {
			Logger.error(e);
			return Collections.emptyList();
		}
	}

	@NotNull
	private GraphServiceClient<Request> client() {
		return GraphServiceClient
				.builder()
				.authenticationProvider(getAuth())
				.buildClient();
	}

	private TokenCredentialAuthProvider getAuth() {
		return new TokenCredentialAuthProvider(new ClientSecretCredentialBuilder()
				.tenantId(credentials.getChannelAuthTenant())
				.clientId(credentials.getAppId())
				.clientSecret(credentials.getAppPassword())
				.build());
	}
}