package io.intino.amidas.connectors.microsoft.teams;

import com.microsoft.bot.connector.authentication.MicrosoftAppCredentials;
import com.microsoft.bot.integration.AdapterWithErrorHandler;
import com.microsoft.bot.integration.BotFrameworkHttpAdapter;
import com.microsoft.bot.integration.Configuration;
import io.intino.alexandria.http.AlexandriaSpark;
import io.intino.alexandria.logger.Logger;
import io.intino.amidas.connectors.microsoft.teams.model.Team.Channel;
import io.intino.amidas.shared.Team;
import io.intino.amidas.shared.connectors.MessagingConnector;

import javax.naming.ConfigurationException;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static io.intino.amidas.connectors.microsoft.teams.ConfigurationKeys.*;

public class TeamsConnector implements MessagingConnector {
	private final Properties configuration;
	private BotController botController;
	private ProxyBot bot;
	private MicrosoftAppCredentials credentials;
	private List<io.intino.amidas.connectors.microsoft.teams.model.Team> teams;
	private Team team;

	public TeamsConnector(Properties configuration) {
		this.configuration = configuration;
	}

	@Override
	public void setup(AlexandriaSpark spark) throws ConfigurationException {
		if (configuration == null || !keys().allMatch(configuration::containsKey))
			throw new ConfigurationException("Expected the next keys in configuration: " + String.join(", ", keys().toArray(String[]::new)));
		BotFrameworkHttpAdapter adapter = new AdapterWithErrorHandler(microsoftConfiguration());
		credentials = credentials();
		bot = new ProxyBot(credentials, configuration, adapter);
		botController = new BotController(spark, adapter, bot);
	}

	@Override
	public void start(Team team) {
		this.team = team;
		scheduleReloadTeams();
		botController.start();
	}

	@Override
	public boolean sendMessage(String user, String message) {
		Team.Identity identity = identity(user);
		if (identity == null) {
			Logger.error("User '" + user + "' not found in identities");
			return false;
		}
		if (message.isEmpty()) {
			Logger.error("Notification empty");
			return false;
		}
		Logger.info("Trying to notify " + message + " to " + user);
		try {
			return bot.sendMessageToUser(identity.get("id/username").value(), message);
		} catch (IllegalArgumentException e) {
			Logger.error(e);
			return false;
		}
	}

	@Override
	public boolean sendMessage(String teamName, String channelName, String message) {
		if (message.isEmpty()) {
			Logger.error("Notification empty");
			return false;
		}
		Logger.info("Trying to notify " + message + " to " + teamName + " -> " + channelName);
		try {
			Channel channel = channel(teamName, channelName);
			return bot.sendMessageToChannel(channel.id(), message);
		} catch (IllegalArgumentException e) {
			Logger.error(e);
			return false;
		}
	}

	private Channel channel(String team, String channel) {
		if (team == null) throw new IllegalArgumentException("Team not found");
		return teams.stream()
				.filter(tm -> tm.displayName().equals(team)).findFirst()
				.orElseThrow(() -> new IllegalArgumentException("Team " + team + " not found"))
				.channels().stream()
				.filter(c -> c.displayName().equals(channel)).findFirst()
				.orElseThrow(() -> new IllegalArgumentException("Channel " + channel + " not found"));
	}

	private Configuration microsoftConfiguration() {
		return new Configuration() {

			private static final Map<String, String> PropertyMapper = Map.of("MicrosoftAppId", "clientId", "MicrosoftAppPassword", "clientSecret", "MicrosoftAppTenant", "tenantId");

			@Override
			public String getProperty(String s) {
				return configuration.getProperty(PropertyMapper.getOrDefault(s, s));
			}

			@Override
			public Properties getProperties() {
				Properties properties = new Properties();
				properties.put("MicrosoftAppId", properties.getProperty(ClientId));
				properties.put("MicrosoftAppPassword", properties.getProperty(ClientSecret));
				properties.put("MicrosoftAppTenant", properties.getProperty(TenantId));
				properties.put("server.port", properties.getProperty(Port));
				return properties;
			}

			@Override
			public String[] getProperties(String s) {
				return getProperties().stringPropertyNames().toArray(new String[0]);
			}
		};
	}

	private MicrosoftAppCredentials credentials() {
		return new MicrosoftAppCredentials(configuration.getProperty(ClientId), configuration.getProperty(ClientSecret), configuration.getProperty(TenantId));
	}

	private void scheduleReloadTeams() {
		ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor();
		executor.scheduleAtFixedRate(() -> {
			List<io.intino.amidas.connectors.microsoft.teams.model.Team> newTeams = new TeamsInfo(credentials).reloadTeams();
			teams.clear();
			teams.addAll(newTeams);
		}, 0, 1, TimeUnit.HOURS);
		teams = new TeamsInfo(credentials).reloadTeams();
	}

	private Team.Identity identity(String user) {
		List<Team.Identity> result = team.search("id/username", user);
		if (result.isEmpty()) result = team.search("id/email", user);
		return !result.isEmpty() ? result.get(0) : null;
	}

}
