package io.intino.amidas.connectors.microsoft.teams;

import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.microsoft.bot.builder.TurnContext;
import com.microsoft.bot.builder.teams.TeamsActivityHandler;
import com.microsoft.bot.connector.authentication.MicrosoftAppCredentials;
import com.microsoft.bot.integration.BotFrameworkHttpAdapter;
import com.microsoft.bot.schema.*;
import io.intino.alexandria.Json;
import io.intino.alexandria.cli.schemas.BotResponse;
import io.intino.alexandria.cli.schemas.BotTalk;
import io.intino.alexandria.exceptions.AlexandriaException;
import io.intino.alexandria.logger.Logger;
import io.intino.alexandria.restaccessor.Response;
import io.intino.alexandria.restaccessor.core.RestAccessor;
import io.intino.alexandria.restaccessor.exceptions.RestfulFailure;
import org.apache.commons.lang3.StringUtils;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.CompletableFuture;

import static com.codepoetics.protonpack.collectors.CompletableFutures.toFutureList;
import static com.microsoft.bot.builder.MessageFactory.suggestedActions;
import static com.microsoft.bot.builder.MessageFactory.text;
import static io.intino.alexandria.cli.schemas.BotResponse.Type.multiline;
import static io.intino.alexandria.cli.schemas.BotResponse.Type.question;
import static io.intino.alexandria.cli.schemas.BotResponse.Type.text;
import static io.intino.amidas.connectors.microsoft.teams.ConfigurationKeys.*;

public class ProxyBot extends TeamsActivityHandler {
	private final MicrosoftAppCredentials credentials;
	private final Properties configuration;
	private final ConversationReferences conversations;
	private final BotFrameworkHttpAdapter adapter;

	public ProxyBot(MicrosoftAppCredentials credentials, Properties configuration, BotFrameworkHttpAdapter adapter) {
		this.credentials = credentials;
		this.configuration = configuration;
		this.adapter = adapter;
		this.conversations = loadConversations();
	}

	public boolean sendMessageToUser(String user, String message) {
		if (conversations.isEmpty()) {
			Logger.error("There aren't conversations");
			return false;
		}
		ConversationReference ref = conversations.values().stream().filter(c -> user.equals(c.getUser().getAadObjectId())).findFirst().orElse(null);
		if (ref == null) return false;
		adapter.continueConversation(credentials.getAppId(), ref, ctx -> ctx.sendActivity(message).thenApply(resourceResponse -> null));
		return true;
	}

	public boolean sendMessageToChannel(String channelId, String message) {
		if (conversations.isEmpty()) {
			Logger.error("There aren't conversations");
			return false;
		}
		ConversationReference conv = conversations.values().stream().filter(c -> !c.getServiceUrl().contains("botframework.com")).findFirst().orElse(conversations.values().iterator().next());
		Logger.info("Sending notification to " + channelId + ": " + message);
		adapter.createConversation(channelId, conv.getServiceUrl(), credentials, notificationParameters(channelId, conv.getBot(), text(message)), null).thenApply(started -> null);
		return true;
	}

	@Override
	protected CompletableFuture<Void> onConversationUpdateActivity(TurnContext context) {
		Logger.info("Updated conversation activity");
		addConversationReference(context.getActivity());
		return super.onConversationUpdateActivity(context);
	}

	@Override
	protected CompletableFuture<Void> onMembersAdded(List<ChannelAccount> membersAdded, TurnContext context) {
		addConversationReference(context.getActivity());
		String welcomeText = "Hello and welcome!";
		return membersAdded.stream()
				.filter(member -> !StringUtils.equals(member.getId(), context.getActivity().getRecipient().getId()))
				.map(channel -> context.sendActivity(text(welcomeText, welcomeText, null)))
				.collect(toFutureList()).thenApply(resourceResponses -> null);
	}

	@Override
	protected CompletableFuture<Void> onMessageActivity(TurnContext context) {
		try {
			addConversationReference(context.getActivity());
			String user = user(context);
			String request = context.getActivity().getText();
			Logger.debug("request: " + request);
			if (isMention(request)) request = cleanMention(request);
			BotResponse response = queryApp(context, user, request);
			if (response.type().equals(text)) return sendTextResponse(context, response);
			if (response.type().equals(question)) return sendQuestionResponse(context, response);
			if (response.type().equals(multiline)) return sendMultilineResponse(context, response);
			return sendFile(response);
		} catch (AlexandriaException | RestfulFailure e) {
			Logger.error(e.getMessage());
			return context.sendActivity("Error processing request: " + e.getMessage()).thenApply(r -> null);
		}
	}

	private CompletableFuture<Void> sendQuestionResponse(TurnContext context, BotResponse response) {
		Activity activity = suggestedActions(response.options(), response.title());
		activity.getSuggestedActions().setTo(List.of(context.getActivity().getFrom().getId()));
		return context.sendActivity(activity).thenApply(sendResult -> null);
	}

	private CompletableFuture<Void> sendTextResponse(TurnContext context, BotResponse botResponse) {
		return context.sendActivity(text(format(botResponse.raw()))).thenApply(result -> null);
	}

	private CompletableFuture<Void> sendMultilineResponse(TurnContext context, BotResponse botResponse) {
		return context.sendActivity(text(format(botResponse.raw()))).thenApply(result -> null);
	}

	private String format(String text) {
		return text.replace("\n", "\n\n")
				.replace(":+1:", "&#x1F44D;")
				.replace(":cpu:", "&#x26CF;")
				.replace(":memory:", "&#x1F4C0;")
				.replace(":small_red_triangle:", "&#x1F53C;")
				.replace(":small_red_triangle_down:", "&#x1F53D;")
				.replace(":hdd:", "&#x1F4BE;")
				.replace(":clock10:", "&#x1F551;")
				.replace(":threads:", "&#x1F9F5;")
				.replace(":network:", "&#x1F310;")
				.replace(":started:", "&#x1F7E2;")
				.replace(":stop:", "&#x1F534;")
				;
	}

	private CompletableFuture<Void> sendFile(BotResponse response) {
		return getInlineAttachment(response)
				.thenApply(attachment -> {
					Activity reply = text("This is an inline attachment.");
					reply.setAttachment(attachment);
					return reply;
				})
				.exceptionally(ex -> text("There was an error handling the attachment: " + ex.getMessage())
				).thenCompose(r -> null);
	}

	private BotResponse queryApp(TurnContext context, String userId, String messageContent) throws AlexandriaException, RestfulFailure {
		io.intino.alexandria.restaccessor.RestAccessor.RestfulSecureConnection restAccessor = new RestAccessor().secure(urlOf(configuration.getProperty(AppUrl)), userId);
		BotTalk talk = new BotTalk().conversation(messageContent).timeZone(context.getActivity().getLocalTimezone());
		Response post = restAccessor.post(configuration.getProperty(ConfigurationKeys.CliPath), Json.toJson(talk));
		return Json.fromJson(post.content(), BotResponse.class);
	}

	private URL urlOf(String url) {
		try {
			return new URI(url).toURL();
		} catch (MalformedURLException | URISyntaxException e) {
			Logger.error(e);
			return null;
		}
	}

	private CompletableFuture<Attachment> getInlineAttachment(BotResponse response) {
		return CompletableFuture.completedFuture(response.raw())
				.thenApply(encodedFileData -> {
					Attachment attachment = new Attachment();
					attachment.setName(response.fileName());
					attachment.setContentType("image/png");
					attachment.setContentUrl("data:image/png;base64," + encodedFileData);
					return attachment;
				});
	}

	private String cleanMention(String request) {
		return request.replace("<at>" + configuration.getProperty(BotName) + "</at>", "").trim();
	}

	private boolean isMention(String request) {
		return request.startsWith("<at>" + configuration.getProperty(BotName) + "</at>");
	}

	private void addConversationReference(Activity activity) {
		ConversationReference ref = activity.getConversationReference();
		conversations.put(ref.getUser().getId(), ref);
		saveConversationReferences();
	}

	private ConversationReferences loadConversations() {
		try {
			Path path = conversationsFile();
			if (!path.toFile().exists()) return new ConversationReferences();
			return Json.fromString(Files.readString(path), ConversationReferences.class);
		} catch (IOException e) {
			Logger.error(e);
			return new ConversationReferences();
		}
	}

	private Path conversationsFile() {
		String storeDir = configuration.getProperty(StoreDirectory);
		new File(storeDir).mkdirs();
		return new File(storeDir, "conversations.json").toPath();
	}


	private static ConversationParameters notificationParameters(String channel, ChannelAccount bot, Activity message) {
		ObjectNode channelData = JsonNodeFactory.instance.objectNode();
		channelData.set("channel", JsonNodeFactory.instance.objectNode().set("id", JsonNodeFactory.instance.textNode(channel)));
		ConversationParameters params = new ConversationParameters();
		params.setIsGroup(true);
		params.setBot(bot);
		params.setActivity(message);
		params.setChannelData(channelData);
		return params;
	}


	private void saveConversationReferences() {
		try {
			Files.writeString(conversationsFile(), String.join("\n", Json.toString(conversations)));
		} catch (IOException e) {
			Logger.error(e);
		}
	}

	private String user(TurnContext context) {
		return context.getActivity().getFrom().getAadObjectId();
	}
}