package io.intino.amidas.connectors.microsoft.azure;

import io.intino.alexandria.http.AlexandriaSpark;
import io.intino.alexandria.logger.Logger;
import io.intino.amidas.connectors.microsoft.azure.model.User;
import io.intino.amidas.shared.Identity;
import io.intino.amidas.shared.Team;
import io.intino.amidas.shared.connectors.ActiveDirectoryConnector;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import static java.util.stream.Collectors.toMap;

public class AzureConnector implements ActiveDirectoryConnector {
	private final OperationMode operationMode;
	private ActiveDirectoryAccessor activeDirectory;
	private final File credentialsFile;
	private Team team;
	private UserFilter filter;

	private static final AtomicBoolean Enabled = new AtomicBoolean(false);

	public AzureConnector(File credentialsFile, OperationMode operationMode) {
		this.credentialsFile = credentialsFile;
		this.operationMode = operationMode;
	}

	@Override
	public void setup(AlexandriaSpark alexandriaSpark) {
	}

	@Override
	public void start(io.intino.amidas.shared.Team team) {
		if (!credentialsFile.exists()) {
			Logger.error("Active Directory credentials file not found");
			return;
		}
		this.team = team;
		this.activeDirectory = new ActiveDirectoryAccessor(credentialsFile);
		Enabled.set(true);
		refresh();
	}

	@Override
	public synchronized void refresh() {
		if (!Enabled.get()) return;
		activeDirectory.refresh();
		List<Team.Identity> identityList = team.search("");
		Map<String, Team.Identity> identityMap = identityList.stream()
				.collect(toMap(i -> i.getId("username").value(), i -> i, (i1, i2) -> i1));
		try {
			activeDirectory.users().stream().filter(this::matches).forEach(user -> operateIdentity(user, identityMap));
		} catch (Exception e) {
			Logger.error(e);
		}
	}

	@Override
	public Frequency refreshFrequency() {
		return Frequency.Hourly;
	}

	@Override
	public OperationMode operationMode() {
		return operationMode;
	}

	@Override
	public void filter(UserFilter filter) {
		this.filter = filter;
	}

	private boolean matches(User user) {
		return filter == null || filter.matches(properties(user));
	}

	private void operateIdentity(User user, Map<String, Team.Identity> identityMap) {
		boolean exists = identityMap.containsKey(user.id);
		OperationMode operationMode = operationMode();
		if (operationMode == OperationMode.CreateIdentity && !exists) createIdentity(user);
		if (operationMode == OperationMode.UpdateIdentity && exists) updateIdentity(user, identityMap.get(user.id));
		if (operationMode == OperationMode.CreateOrUpdateIdentity) {
			if (exists) updateIdentity(user, identityMap.get(user.id));
			else createIdentity(user);
		}
	}

	private void createIdentity(User user) {
		try {
			Team.Identity identity = team.createIdentity();
			identity.append("id/username", user.id);
			identity.append("id/email", user.mail);
			identity.append("feature/fullName", user.displayName);
			identity.append("feature/language", user.language != null ? user.language : "en");
			identity.save();
		} catch (IOException e) {
			Logger.error(e);
		}
	}

	private void updateIdentity(User user, Team.Identity identity) {
		try {
			Team.Builder builder = team.build(identity);
			boolean hasChanges = false;
			if (!user.id.equals(valueOf(identity.getId("username")))) { identity.append("id/username", user.id); hasChanges = true; }
			if (user.mail != null && !user.mail.equals(valueOf(identity.getId("email")))) { identity.append("id/email", user.mail); hasChanges = true; }
			if (user.displayName != null && !user.displayName.equals(valueOf(identity.getFeature("fullName")))) { identity.append("feature/fullName", user.displayName); hasChanges = true; }
			if (user.language != null && !user.language.equals(valueOf(identity.getFeature("language")))) { identity.append("feature/language", user.language); hasChanges = true; }
			if (hasChanges) builder.save();
		} catch (IOException e) {
			Logger.error(e);
		}
	}

	private String valueOf(Team.Id id) {
		return id != null ? id.value() : null;
	}

	private String valueOf(Team.Feature feature) {
		return feature != null ? feature.value() : null;
	}

	private UserFilter.UserInfo properties(User user) {
		return new UserFilter.UserInfo() {
			@Override
			public String id() {
				return user.id;
			}

			@Override
			public String email() {
				return user.mail;
			}

			@Override
			public String language() {
				return user.language;
			}

			@Override
			public String attribute(String name) {
				if (name.equalsIgnoreCase("fullName")) return user.displayName;
				return null;
			}
		};
	}

}