package io.intino.amidas.identityeditor.box.ui.displays.templates;

import io.intino.alexandria.core.Box;
import io.intino.alexandria.ui.displays.UserMessage;
import io.intino.amidas.identityeditor.box.ContextManager;
import io.intino.amidas.shared.Team;

import java.text.NumberFormat;
import java.util.Collections;
import java.util.List;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

public class TeamEditor extends AbstractTeamEditor<Box> {
    private Team team;
    private String condition;
    private int current = 0;
    private List<Team.Identity> identities = Collections.emptyList();
    private Consumer<Team> modifiedListener;
    private BiConsumer<Team.Identity, List<Team.Statement>> identityModifiedListener;

    public TeamEditor(Box box) {
        super(box);
    }

    public TeamEditor team(Team team) {
        this.team = team;
        return this;
    }

    public TeamEditor onModify(Consumer<Team> listener) {
        this.modifiedListener = listener;
        return this;
    }

    public TeamEditor onModifyIdentity(BiConsumer<Team.Identity, List<Team.Statement>> listener) {
        this.identityModifiedListener = listener;
        return this;
    }

    public void filter(String condition) {
        this.condition = condition;
        identities = team != null ? team.search(condition) : Collections.emptyList();
        current = 0;
    }

    @Override
    public void init() {
        super.init();
        initToolbar();
        initDialog();
    }

    private void initToolbar() {
        search.onChange(e -> {
            filter(e.value());
            refresh();
        });
        search.onEnterPress(e -> filter());
        previous.onExecute(e -> previous());
        next.onExecute(e -> next());
        // TODO TEAM NO PERMITE ELIMINAR IDENTIDADES removeIdentity.onExecute(e -> removeIdentity());
    }

    private void initDialog() {
        openAddIdentity.onOpen(e -> {
            identityDialogBox.title(translate("Add identity"));
            identityDialogBox.main.dialog.properties(team.grammar().getAll());
            identityDialogBox.main.dialog.identity(createIdentity());
            identityDialogBox.main.dialog.refresh();
        });
        openEditIdentity.onOpen(e -> {
            identityDialogBox.title(translate("Edit identity"));
            identityDialogBox.main.dialog.properties(team.grammar().getAll());
            identityDialogBox.main.dialog.identity(current());
            identityDialogBox.main.dialog.refresh();
        });
        cancel.onExecute(e -> identityDialogBox.close());
        accept.onExecute(e -> saveIdentity());
    }

    private Team.Identity createIdentity() {
        return team.createIdentity();
    }

    private void removeIdentity() {
        ContextManager.register(soul());
        // TEAM no permite eliminar identidades
        notifyUser(translate("Identity removed"), UserMessage.Type.Success);
        notifySaveIdentities();
        previous();
    }

    private void saveIdentity() {
        identityDialogBox.close();
        ContextManager.register(soul());
        notifyUser(translate("Identity saved"), UserMessage.Type.Success);
        notifySave(identityDialogBox.main.dialog.identity(), identityDialogBox.main.dialog.statements());
        refreshIdentity(identityDialogBox.main.dialog.identity());
    }

    private void notifySaveIdentities() {
        if (modifiedListener != null) modifiedListener.accept(team);
    }

    private void notifySave(Team.Identity identity, List<Team.Statement> statements) {
        notifySaveIdentities();
        if (identityModifiedListener != null) identityModifiedListener.accept(identity, statements);
    }

    public void reloadTeam(Team team) {
        this.team = team;
        identities = team.search(condition);
        updateIdentity();
        refresh();
    }

    private void updateIdentity() {
        Team.Identity identity = identityDialogBox.main.dialog.identity();
        if (identity == null) return;
        int pos = findIdentity(identity);
        this.current = pos != -1 ? pos : 0;
    }

    private int findIdentity(Team.Identity identity) {
        for (int i=0; i<identities.size(); i++) {
            if (identities.get(i).id().equals(identity.id())) return i;
        }
        return -1;
    }

    @Override
    public void refresh() {
        super.refresh();
        refreshLayers();
        if (team == null) return;
        refreshToolbar();
        refreshIdentity(current());
        refreshInfo();
    }

    private void refreshLayers() {
        identitiesFileNotFoundLayer.visible(team == null);
        teamEditorLayer.visible(team != null);
    }

    private void refreshToolbar() {
        previous.readonly(!canPrevious());
        next.readonly(!canNext());
    }

    private void refreshIdentity(Team.Identity identity) {
        identityViewer.identity(identity, team.grammar().getAll());
        identityViewer.refresh();
    }

    private void refreshInfo() {
        String message = condition != null && !condition.isEmpty() ? "%s " + translate("for") + " %s" : "%s";
        info.value(String.format(translate(message), identitiesCountMessage(identities.size()), condition));
    }

    private Team.Identity current() {
        return identities.size() > 0 ? identities.get(current) : null;
    }

    private void previous() {
        current--;
        if (current < 0) current = 0;
        refresh();
    }

    private boolean canPrevious() {
        return identities.size() > 0 && current > 0;
    }

    private void next() {
        current++;
        if (current >= identities.size()) current = identities.size()-1;
        refresh();
    }

    private boolean canNext() {
        return identities.size() > 0 && current < identities.size()-1;
    }

    private void filter() {
        filter(condition);
        refresh();
    }

    public String identitiesCountMessage(int count) {
        if (count <= 0) return translate("No identities");
        if (count == 1) return translate("1 identity");
        return NumberFormat.getNumberInstance().format(count) + translate(" identities");
    }

}