package io.intino.amidas.accessor.alexandria.core;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonPrimitive;
import io.intino.alexandria.restaccessor.Response;
import io.intino.alexandria.restaccessor.RestAccessor;
import io.intino.alexandria.restaccessor.exceptions.RestfulFailure;
import io.intino.alexandria.ui.services.AuthService;
import io.intino.alexandria.ui.services.auth.FederationInfo;
import io.intino.alexandria.ui.services.auth.Space;
import io.intino.alexandria.ui.services.auth.Token;
import io.intino.alexandria.ui.services.auth.UserInfo;
import io.intino.alexandria.ui.services.auth.exceptions.*;
import io.intino.amidas.accessor.alexandria.adapters.response.FederationInfoResponseAdapter;
import io.intino.amidas.accessor.alexandria.adapters.response.UserInfoResponseAdapter;
import io.intino.amidas.accessor.core.KonosPushClient;
import io.intino.amidas.accessor.core.Verifier;
import org.java_websocket.client.WebSocketClient;
import org.java_websocket.handshake.ServerHandshake;
import org.scribe.builder.ServiceBuilder;
import org.scribe.builder.api.DefaultApi10a;
import org.scribe.oauth.OAuthService;

import java.net.URL;
import java.util.*;

public class AmidasOauthAccessor implements AuthService {
    private final Space space;
    private final URL amidasUrl;
    private Map<URL, KonosPushClient> pushClientMap = new HashMap<>();
    private RestAccessor api;

    public AmidasOauthAccessor(Space space, URL amidasUrl) {
        this.space = space;
        this.amidasUrl = amidasUrl;
        this.api = new io.intino.alexandria.restaccessor.core.RestAccessor();
    }

    @Override
    public URL url() {
        return amidasUrl;
    }

    @Override
    public Space space() {
        return space;
    }

    @Override
    public Authentication authenticate() throws SpaceAuthCallbackUrlIsNull {
        return new Authentication() {
            private OAuthService authService = serviceOf(amidasUrl, space);
            private Token requestToken;
            private Token accessToken;

            @Override
            public Token requestToken() throws CouldNotObtainRequestToken {

                try {
                    this.requestToken = tokenFrom(Optional.of(authService.getRequestToken()));
                } catch (Exception exception) {
                    throw new CouldNotObtainRequestToken(exception);
                }

                this.accessToken = null;
                return this.requestToken;
            }

            @Override
            public URL authenticationUrl(Token requestToken) throws CouldNotObtainAuthorizationUrl {
                if (this.requestToken != requestToken)
                    return null;

                try {
                    return new URL(authService.getAuthorizationUrl(token(Optional.of(requestToken))));
                } catch (Exception exception) {
                    throw new CouldNotObtainAuthorizationUrl(exception);
                }
            }

            @Override
            public Token accessToken() {
                return accessToken;
            }

            @Override
            public Token accessToken(io.intino.alexandria.ui.services.auth.Verifier verifier) throws CouldNotObtainAccessToken {
                if (requestToken == null)
                    return null;

                try {
                    org.scribe.model.Token accessToken = authService.getAccessToken(token(Optional.of(requestToken)), verifier(verifier));
                    this.accessToken = tokenFrom(Optional.of(accessToken));
                } catch (Exception exception) {
                    throw new CouldNotObtainAccessToken(exception);
                }

                return this.accessToken;
            }

            @Override
            public void invalidate() throws CouldNotInvalidateAccessToken {
                try {
                    api.post(amidasUrl, String.format("/api/invalidate/%s", accessToken.id()));
                } catch (Exception exception) {
                    throw new CouldNotInvalidateAccessToken(exception);
                }
            }

            @Override
            public Version version() {
                return Version.OAuth1;
            }
        };
    }

    @Override
    public boolean valid(Token accessToken) {
        try {
            if (accessToken == null) return false;
            return getAndCheck(amidasUrl, String.format("/api/valid/%s", accessToken.id()));
        } catch (Exception e) {
            return false;
        }
    }

    @Override
    public FederationInfo info(Token accessToken) throws CouldNotObtainInfo {
        try {
            Response response = api.get(amidasUrl, String.format("/api/info/%s", accessToken.id()));
            return new FederationInfoResponseAdapter().adapt(response.content());
        }
        catch (Exception exception) {
            throw new CouldNotObtainInfo(exception);
        }
    }

    @Override
    public UserInfo me(Token accessToken) throws CouldNotObtainInfo {
        try {
            Response response = api.get(amidasUrl, String.format("/api/me/%s", accessToken.id()));
            return new UserInfoResponseAdapter().adapt(response.content());
        } catch (RestfulFailure exception) {
            throw new CouldNotObtainInfo(exception);
        }
    }

    @Override
    public void logout(Token accessToken) throws CouldNotLogout {
        try {
            api.post(amidasUrl, String.format("/api/logout/%s", accessToken.id()));
        } catch (Exception exception) {
            throw new CouldNotLogout(exception);
        }
    }

    @Override
    public String logoutUrl() {
        return null;
    }

    @Override
    public void addPushListener(Token accessToken, FederationNotificationListener listener) throws CouldNotObtainInfo {

        if (!pushClientMap.containsKey(amidasUrl)) {
            FederationInfo info = info(accessToken);
            pushClientMap.put(amidasUrl, createNotificationClient(info));
        }

        KonosPushClient pushClient = pushClientMap.get(amidasUrl);
        pushClient.addListener(listener);
    }

    private KonosPushClient createNotificationClient(FederationInfo info) {
        return new AmidasPushClient(info);
    }

    private Token tokenFrom(Optional<org.scribe.model.Token> token) {
        if (!token.isPresent())
            return null;

        return new Token() {
            @Override
            public String id() {
                return token.get().getToken();
            }

            @Override
            public String secret() {
                return token.get().getSecret();
            }
        };
    }

    private org.scribe.model.Token token(Optional<Token> token) {
        if (!token.isPresent())
            return null;

        return new org.scribe.model.Token(token.get().id(), "");
    }

    private org.scribe.model.Verifier verifier(io.intino.alexandria.ui.services.auth.Verifier verifier) {
        return new org.scribe.model.Verifier(verifier.value());
    }

    private OAuthService serviceOf(URL federation, Space space) throws SpaceAuthCallbackUrlIsNull {
        ServiceBuilder builder = new ServiceBuilder().provider(apiOf(federation)).apiKey(space.name()).apiSecret(space.secret());
        URL callbackUrl = space.authCallbackUrl();

        if (callbackUrl == null)
            throw new SpaceAuthCallbackUrlIsNull();

        builder.callback(callbackUrl.toString());

        return builder.build();
    }

    private org.scribe.builder.api.Api apiOf(URL federation) {
        final String AuthenticationPath = "/authentication/%s";
        final String RequestTokenPath = "/authentication/token/request";
        final String AccessTokenPath = "/authentication/token/access";
        final String url = federation.toString();

        return new DefaultApi10a() {
            @Override
            public String getRequestTokenEndpoint() {
                return url + RequestTokenPath;
            }

            @Override
            public String getAccessTokenEndpoint() {
                return url + AccessTokenPath;
            }

            @Override
            public String getAuthorizationUrl(org.scribe.model.Token token) {
                return url + String.format(AuthenticationPath, token.getToken());
            }
        };
    }

    private static class AmidasPushClient implements KonosPushClient {
        private final FederationInfo info;
        private WebSocketClient client;
        private List<FederationNotificationListener> listenerList;

        public AmidasPushClient(FederationInfo info) {
            this.info = info;
            this.listenerList = new ArrayList<>();
            //createClient();
        }

        private void createClient() {
            client = new WebSocketClient(info.pushServerUri()) {
                @Override
                public void onOpen(ServerHandshake serverHandshake) {
                }

                @Override
                public void onMessage(String message) {
                    AmidasOauthAccessor.AmidasPushClient.this.notify(message);
                }

                @Override
                public void onClose(int i, String s, boolean b) {
                }

                @Override
                public void onError(Exception e) {
                }
            };
            client.connect();
        }

        @Override
        public void notify(String rawMessage) {
            Message message = messageOf(rawMessage);

            if (!(rawMessage.contains("userLoggedOut") || rawMessage.contains("userAdded")))
                return;

            for (FederationNotificationListener listener : listenerList) {
                if (message.name.equals("userLoggedOut"))
                    listener.userLoggedOut(new UserInfoResponseAdapter().adapt(message.param("userInfo")));

                if (message.name.equals("userAdded"))
                    listener.userAdded(new UserInfoResponseAdapter().adapt(message.param("userInfo")));
            }
        }

        private Message messageOf(String rawMessage) {
            return Message.build(rawMessage);
        }

        @Override
        public void addListener(FederationNotificationListener listener) {
            listenerList.add(listener);
        }

        private static class Message {
            private String name;
            private JsonObject rawParameters;

            public Message(String name) {
                this.name = name;
            }

            public String name() {
                return name;
            }

            public JsonElement param(String name) {
                if (rawParameters == null)
                    return null;

                return rawParameters.get(name);
            }

            public static Message build(String rawData) {
                JsonElement raw = (new JsonParser()).parse(rawData);

                if (raw instanceof JsonPrimitive)
                    return new Message(raw.getAsString());

                JsonObject rawMessage = (JsonObject) raw;
                Message message = new Message(rawMessage.get("name").getAsString());
                message.rawParameters = rawMessage.get("parameters").getAsJsonObject();

                return message;
            }
        }
    }

    private boolean getAndCheck(URL url, String resource) throws RestfulFailure {
        return Boolean.valueOf(api.get(url, resource).content());
    }

}
