package io.intino.amidas.accessor.alexandria.core;

import com.google.gson.Gson;
import io.intino.alexandria.logger.Logger;
import io.intino.alexandria.ui.services.AuthService;
import io.intino.alexandria.ui.services.auth.*;
import io.intino.alexandria.ui.services.auth.exceptions.CouldNotObtainAuthorizationUrl;
import io.intino.alexandria.ui.services.auth.exceptions.CouldNotObtainInfo;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.ssl.SSLContextBuilder;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.*;

public class AmidasCasAccessor implements AuthService {
	private URL authServiceUrl;
	private Space space;

	public AmidasCasAccessor(Space space, URL authServiceUrl) {
		this.space = space;
		this.authServiceUrl = authServiceUrl;
	}

	@Override
	public URL url() {
		return authServiceUrl;
	}

	@Override
	public Space space() {
		return space;
	}

	@Override
	public Authentication authenticate() {
		return new Authentication() {

			private Token requestToken;
			private Token accessToken;

			@Override
			public Token requestToken() {

				this.requestToken = new Token() {
					@Override
					public String id() {
						return UUID.randomUUID().toString();
					}

					@Override
					public String secret() {
						return "";
					}
				};
				this.accessToken = null;
				return this.requestToken;
			}

			@Override
			public URL authenticationUrl(Token token) throws CouldNotObtainAuthorizationUrl {
				if (this.requestToken != token) {
					return null;
				} else {
					return AmidasCasAccessor.this.authorizationUrl(requestToken.id());
				}
			}

			@Override
			public Token accessToken() {
				return this.accessToken;
			}

			@Override
			public Token accessToken(Verifier verifier) {
				accessToken = verifier != null && accessToken == null ? new Token() {
					@Override
					public String id() {
						return verifier.value();
					}

					@Override
					public String secret() {
						return "1234";
					}
				} : null;
				return this.accessToken;
			}

			@Override
			public void invalidate() {
				logout(accessToken);
			}

			@Override
			public Version version() {
				return Version.OAuth2;
			}
		};
	}

	@Override
	public boolean valid(Token token) {
		try {
			if (token == null || token.id() == null) return false;
			return me(token) != null;
		} catch (CouldNotObtainInfo ignored) {
			return false;
		}
	}

	@Override
	public FederationInfo info(Token token) throws CouldNotObtainInfo {

		return new FederationInfo() {
			@Override
			public String name() {
				return "CAS";
			}

			@Override
			public String title() {
				return "CAS federation";
			}

			@Override
			public String subtitle() {
				return null;
			}

			@Override
			public URL logo() {
				return null;
			}

			@Override
			public URI pushServerUri() {
				return null;
			}
		};
	}

	@Override
	public UserInfo me(Token token) throws CouldNotObtainInfo {
		String tokenId = token.id();
		if (tokenId == null) return null;
		HttpGet get = new HttpGet(url() + "/serviceValidate?ticket=" + tokenId + "&service=" + callbackUrl(space) + "&format=JSON");
		try {
			CloseableHttpClient client = HttpClientFactory.client();
			HttpResponse response = client.execute(get);
			if (response.getStatusLine().getStatusCode() != 200) throw new CouldNotObtainInfo(new Exception("Status!=200"));
			BufferedReader rd = new BufferedReader(new InputStreamReader(response.getEntity().getContent()));
			StringBuilder result = new StringBuilder();
			String line;
			while ((line = rd.readLine()) != null) result.append(line);
			Map<String, Object> map = new Gson().fromJson(result.toString(), Map.class);
			if (!map.containsKey("serviceResponse")) return null;
			Map<String, Object> serviceResponse = (Map<String, Object>) map.get("serviceResponse");
			if (!serviceResponse.containsKey("authenticationSuccess")) return null;
			return userInfo((Map<String, Object>) serviceResponse.get("authenticationSuccess"));
		} catch (IOException e) {
			throw new CouldNotObtainInfo(e);
		}
	}

	@Override
	public void logout(Token token) {
	}

	@Override
	public String logoutUrl() {
		return url() + "/logout";
	}

	@Override
	public void addPushListener(Token token, FederationNotificationListener federationNotificationListener) throws CouldNotObtainInfo {
	}

	private Space createSpace(URL spaceUrl) {
		if (spaceUrl == null) return null;
		Space space = new Space(url());
		space.setBaseUrl(spaceUrl.toString());
		return space;
	}

	private URL urlOf(String url) {
		try {
			return new URL(url);
		} catch (MalformedURLException e) {
			Logger.error(e);
			return null;
		}
	}

	private static final String AuthorizationPath = "/login?service=%s";
	private URL authorizationUrl(String authId) {
		return urlOf(authServiceUrl + String.format(AuthorizationPath, callbackUrl(space)));
	}

	private String callbackUrl(Space space) {
		return encode(space().url().toString() + "/authenticate-callback");
	}

	public static String encode(String value) {
		if (value == null) return null;
		return URLEncoder.encode(value, StandardCharsets.UTF_8);
	}

	private UserInfo userInfo(Map<String, Object> map) {
		return new UserInfo() {
			@Override
			public String username() {
				return map.get("user").toString();
			}

			@Override
			public String fullName() {
				return map.get("user").toString();
			}

			@Override
			public URL photo() {
				return null;
			}

			@Override
			public String email() {
				return "";
			}

			@Override
			public String language() {
				return "es";
			}

			@Override
			public List<String> roleList() {
				return Collections.emptyList();
			}
		};
	}

	public static class HttpClientFactory {

		public static CloseableHttpClient client() throws IOException {
			try {
				SSLContextBuilder builder = new SSLContextBuilder();
				builder.loadTrustMaterial(null, new TrustSelfSignedStrategy());
				SSLConnectionSocketFactory sslConnectionSocketFactory = new SSLConnectionSocketFactory(builder.build(), NoopHostnameVerifier.INSTANCE);
				Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory>create()
						.register("http", new PlainConnectionSocketFactory())
						.register("https", sslConnectionSocketFactory)
						.build();

				PoolingHttpClientConnectionManager cm = new PoolingHttpClientConnectionManager(registry);
				cm.setMaxTotal(100);
				return HttpClients.custom()
						.setSSLSocketFactory(sslConnectionSocketFactory)
						.setConnectionManager(cm)
						.build();
			} catch (NoSuchAlgorithmException | KeyStoreException | KeyManagementException e) {
				throw new IOException("Error getting client");
			}
		}
	}

}
