package io.intino.amidas.accessor.core;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonPrimitive;
import io.intino.alexandria.restaccessor.Response;
import io.intino.alexandria.restaccessor.RestAccessor;
import io.intino.alexandria.restaccessor.exceptions.RestfulFailure;
import io.intino.alexandria.ui.services.auth.Token;
import io.intino.alexandria.ui.services.auth.UserInfo;
import io.intino.alexandria.ui.services.auth.exceptions.CouldNotObtainAccessToken;
import io.intino.alexandria.ui.services.auth.exceptions.CouldNotObtainAuthorizationUrl;
import io.intino.alexandria.ui.services.auth.exceptions.CouldNotObtainRequestToken;
import io.intino.amidas.accessor.AmidasApi;
import io.intino.amidas.accessor.AmidasSetupApi;
import io.intino.amidas.accessor.adapters.request.ParameterRequestAdapter;
import io.intino.amidas.accessor.adapters.response.AmidasInfoResponseAdapter;
import io.intino.amidas.accessor.adapters.response.UserInfoResponseAdapter;
import io.intino.amidas.accessor.core.exceptions.AmidasApiFailure;
import io.intino.amidas.accessor.core.exceptions.SpaceAuthenticateCallbackUrlIsNull;
import org.java_websocket.client.WebSocketClient;
import org.java_websocket.handshake.ServerHandshake;
import org.scribe.builder.ServiceBuilder;
import org.scribe.builder.api.DefaultApi10a;
import org.scribe.oauth.OAuthService;

import java.net.URL;
import java.util.*;

public class AmidasProxy implements AmidasApi, AmidasSetupApi {
    private Map<Amidas, PushClient> pushClientMap = new HashMap<>();
    private RestAccessor api;

    public AmidasProxy() {
        this.api = new io.intino.alexandria.restaccessor.core.RestAccessor();
    }

    private static final String ResponseOk = "OK";

    @Override
    public Authenticating authenticate(Amidas amidas) {
        return space -> new Authenticating.Authentication() {
            private OAuthService authService = serviceOf(amidas, space);
            private Token requestToken;
            private Token accessToken;

            @Override
            public Token requestToken() throws CouldNotObtainRequestToken {

                try {
                    this.requestToken = tokenFrom(Optional.of(authService.getRequestToken()));
                }
                catch(Exception exception) {
                    throw new CouldNotObtainRequestToken(exception);
                }

                this.accessToken = null;
                return this.requestToken;
            }

            @Override
            public URL authenticationUrl(Token requestToken) throws CouldNotObtainAuthorizationUrl {
                if (this.requestToken != requestToken)
                    return null;

                try {
                    return new URL(authService.getAuthorizationUrl(token(Optional.of(requestToken))));
                } catch (Exception e) {
                    throw new CouldNotObtainAuthorizationUrl(e);
                }
            }

            @Override
            public Token accessToken() {
                return accessToken;
            }

            @Override
            public Token accessToken(Verifier verifier) throws CouldNotObtainAccessToken {
                if (requestToken == null)
                    return null;

                try {
                    org.scribe.model.Token accessToken = authService.getAccessToken(token(Optional.of(requestToken)), verifier(verifier));
                    this.accessToken = tokenFrom(Optional.of(accessToken));
                }
                catch(Exception exception) {
                    throw new CouldNotObtainAccessToken(exception);
                }

                return this.accessToken;
            }

            @Override
            public void invalidate() throws AmidasApiFailure {
                try {
                    api.post(amidas.url(), String.format("/api/invalidate/%s", accessToken.id()));
                } catch (Exception e) {
                    throw new AmidasApiFailure(String.format("Could not invalidate token %s", accessToken.id()));
                }
            }
        };
    }

    @Override
    public boolean valid(Amidas amidas, Token accessToken) throws AmidasApiFailure {
        try {
            return getAndCheck(amidas.url(), String.format("/api/valid/%s", accessToken.id()));
        } catch (Exception e) {
            return false;
        }
    }

    @Override
    public AmidasInfo info(Amidas amidas, Token accessToken) throws AmidasApiFailure {
        try {
            Response response = api.get(amidas.url(), String.format("/api/info/%s", accessToken.id()));
            return new AmidasInfoResponseAdapter().adapt(response.content());
        }
        catch (Exception exception) {
            throw new AmidasApiFailure(String.format("Could not receive amidas info with token %s", accessToken.id()));
        }
    }

    @Override
    public UserInfo me(Amidas amidas, Token accessToken) throws AmidasApiFailure {
        try {
            Response response = api.get(amidas.url(), String.format("/api/me/%s", accessToken.id()));
            return new UserInfoResponseAdapter().adapt(response.content());
        } catch (RestfulFailure failure) {
            throw new AmidasApiFailure(String.format("Could not receive user info with token %s", accessToken.id()));
        }
    }

    @Override
    public void logout(Amidas amidas, Token accessToken) throws AmidasApiFailure {
        try {
            api.post(amidas.url(), String.format("/api/logout/%s", accessToken.id()));
        } catch (Exception e) {
            throw new AmidasApiFailure(String.format("Could not logout with token %s", accessToken.id()));
        }
    }

    @Override
    public void addPushListener(Amidas amidas, Token accessToken, AmidasPushListener listener) throws AmidasApiFailure {

        if (!pushClientMap.containsKey(amidas)) {
            AmidasInfo info = info(amidas, accessToken);
            pushClientMap.put(amidas, createPushClient(info));
        }

        PushClient pushClient = pushClientMap.get(amidas);
        pushClient.addListener(listener);
    }

    @Override
    public boolean validate(Amidas amidas, String signature, String hash) throws AmidasApiFailure {
        try {
            Response response = api.post(amidas.url(), "/signature", new HashMap<String, String>() {{
                put("operation", "validate");
                put("signature", signature);
                put("hash", hash);
            }});
            return response.content().equals(ResponseOk);
        } catch (Exception e) {
            throw new AmidasApiFailure("Could not validate signature");
        }
    }

    @Override
    public Connection connect(Amidas amidas) {
        return connect(amidas, null, null);
    }

    @Override
    public Connection connect(Amidas amidas, URL certificate, String password) {
        return new Connection() {
            private RestAccessor.RestfulSecureConnection connection = null;

            @Override
            public boolean ping() {
                try {
                    Response response = secure().get("/ping");
                    return response.content().equals(ResponseOk);
                } catch (Exception e) {
                    return false;
                }
            }

            @Override
            public AmidasInfo info() throws AmidasApiFailure {
                try {
                    Response response = secure().get("/info");
                    return new AmidasInfoResponseAdapter().adapt(response.content());
                } catch (Exception e) {
                    throw new AmidasApiFailure("Could not load amidas info");
                }
            }

            @Override
            public UserInfo user(String username, String email) throws AmidasApiFailure {
                try {
                    Response response = secure().get("/user/info", new HashMap<String, String>() {{
                        put("username", username);
                        put("email", email);
                    }});
                    return new UserInfoResponseAdapter().adapt(response.content());
                } catch (Exception e) {
                    throw new AmidasApiFailure(String.format("Could not load user info of %s with email %s", username, email));
                }
            }

            @Override
            public UserInfo user(Token token) throws AmidasApiFailure {
                try {
                    Response response = secure().get("/user/with-token", new HashMap<String, String>() {{
                        put("token", token.id());
                    }});
                    return new UserInfoResponseAdapter().adapt(response.content());
                } catch (Exception e) {
                    throw new AmidasApiFailure(String.format("Could not load user info from token %s", token));
                }
            }

            @Override
            public UserInfo user(String authentication, List<Parameter> parameters) throws AmidasApiFailure {
                try {
                    Response response = secure().post("/user/with-credential", new HashMap<String, String>() {{
                        put("authentication", authentication);
                        put("parameters", new ParameterRequestAdapter().adaptList(parameters).toString());
                    }});
                    return new UserInfoResponseAdapter().adapt(response.content());
                } catch (Exception e) {
                    throw new AmidasApiFailure(String.format("Could not validate user for authentication %s", authentication));
                }
            }

            private RestAccessor.RestfulSecureConnection secure() {
                if (connection == null)
                    connection = api.secure(amidas.url(), certificate, password);
                return connection;
            }
        };
    }

    private PushClient createPushClient(AmidasInfo info) {
        return new AmidasPushClient(info);
    }

    private Token tokenFrom(Optional<org.scribe.model.Token> token) {
        if (!token.isPresent())
            return null;

        return new Token() {
            @Override
            public String id() {
                return token.get().getToken();
            }

            @Override
            public String secret() {
                return token.get().getSecret();
            }
        };
    }

    private org.scribe.model.Token token(Optional<Token> token) {
        if (!token.isPresent())
            return null;

        return new org.scribe.model.Token(token.get().id(), "");
    }

    private org.scribe.model.Verifier verifier(Verifier verifier) {
        return new org.scribe.model.Verifier(verifier.value());
    }

    private OAuthService serviceOf(Amidas amidas, Space space) throws SpaceAuthenticateCallbackUrlIsNull {
        ServiceBuilder builder = new ServiceBuilder().provider(apiOf(amidas)).apiKey(space.name()).apiSecret(space.secret());
        URL callbackUrl = space.authenticateCallbackUrl();

        if (callbackUrl == null)
            throw new SpaceAuthenticateCallbackUrlIsNull();

        builder.callback(callbackUrl.toString());

        return builder.build();
    }

    private org.scribe.builder.api.Api apiOf(Amidas amidas) {
        final String AuthenticationPath = "/authentication/%s";
        final String RequestTokenPath = "/authentication/token/request";
        final String AccessTokenPath = "/authentication/token/access";
        final String url = amidas.url().toString();

        return new DefaultApi10a() {
            @Override
            public String getRequestTokenEndpoint() {
                return url + RequestTokenPath;
            }

            @Override
            public String getAccessTokenEndpoint() {
                return url + AccessTokenPath;
            }

            @Override
            public String getAuthorizationUrl(org.scribe.model.Token token) {
                return url + String.format(AuthenticationPath, token.getToken());
            }
        };
    }

    private static class AmidasPushClient implements PushClient {
        private final AmidasInfo info;
        private WebSocketClient client;
        private List<AmidasPushListener> listenerList;

        public AmidasPushClient(AmidasInfo info) {
            this.info = info;
            this.listenerList = new ArrayList<>();
            createClient();
        }

        private void createClient() {
            client = new WebSocketClient(info.pushServerUri()) {
                @Override
                public void onOpen(ServerHandshake serverHandshake) {
                    System.out.println("connection opened with server");
                }

                @Override
                public void onMessage(String message) {
                    AmidasPushClient.this.notify(message);
                }

                @Override
                public void onClose(int i, String s, boolean b) {
                }

                @Override
                public void onError(Exception e) {
                }
            };
            client.connect();
        }

        @Override
        public void notify(String rawMessage) {
            Message message = messageOf(rawMessage);

            if (!(rawMessage.contains("userLoggedOut") || rawMessage.contains("userAdded")))
                return;

            for (AmidasPushListener listener : listenerList) {
                if (message.name.equals("userLoggedOut"))
                    listener.userLoggedOut(new UserInfoResponseAdapter().adapt(message.param("userInfo")));
            }
        }

        private Message messageOf(String rawMessage) {
            return Message.build(rawMessage);
        }

        @Override
        public void addListener(AmidasPushListener listener) {
            listenerList.add(listener);
        }

        private static class Message {
            private String name;
            private JsonObject rawParameters;

            public Message(String name) {
                this.name = name;
            }

            public String name() {
                return name;
            }

            public JsonElement param(String name) {
                if (rawParameters == null)
                    return null;

                return rawParameters.get(name);
            }

            public static Message build(String rawData) {
                JsonElement raw = (new JsonParser()).parse(rawData);

                if (raw instanceof JsonPrimitive)
                    return new Message(raw.getAsString());

                JsonObject rawMessage = (JsonObject) raw;
                Message message = new Message(rawMessage.get("name").getAsString());
                message.rawParameters = rawMessage.get("parameters").getAsJsonObject();

                return message;
            }
        }
    }

    private boolean getAndCheck(URL url, String resource) throws RestfulFailure {
        return Boolean.valueOf(api.get(url, resource).content());
    }

}
