package io.intino.amidas.accessor.alexandria.core;

import com.google.gson.Gson;
import io.intino.alexandria.logger.Logger;
import io.intino.alexandria.restaccessor.Response;
import io.intino.alexandria.restaccessor.RestAccessor;
import io.intino.alexandria.ui.services.AuthService;
import io.intino.alexandria.ui.services.auth.*;
import io.intino.alexandria.ui.services.auth.exceptions.*;
import io.intino.amidas.accessor.core.Configuration;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.ssl.SSLContextBuilder;
import org.scribe.builder.ServiceBuilder;
import org.scribe.builder.api.Api;
import org.scribe.builder.api.DefaultApi20;
import org.scribe.model.OAuthConfig;
import org.scribe.oauth.OAuthService;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.*;

public class AmidasAzureAccessor implements AuthService {
	private final Space space;
	private final Configuration configuration;
	private UserInfo userInfo;
	private final RestAccessor api;

	private static final String GraphUrl = "graphUrl";
	private static final String TenantId = "tenantId";
	private static final String ClientId = "clientId";
	private static final String ClientSecret = "clientSecret";

	public AmidasAzureAccessor(Space space, Configuration configuration) {
		this.space = space;
		this.configuration = configuration;
		this.api = new io.intino.alexandria.restaccessor.core.RestAccessor();
	}

	@Override
	public URL url() {
		return configuration.url();
	}

	@Override
	public Space space() {
		return space;
	}

	@Override
	public Authentication authenticate() throws SpaceAuthCallbackUrlIsNull {
		return new Authentication() {
			private final OAuthService authService = authService();
			private Token requestToken;
			private Token accessToken;

			@Override
			public Token requestToken() {
				this.requestToken = new Token() {
					@Override
					public String id() {
						return UUID.randomUUID().toString();
					}

					@Override
					public String secret() {
						return "";
					}
				};
				this.accessToken = null;
				return this.requestToken;
			}

			@Override
			public URL authenticationUrl(Token token) throws CouldNotObtainAuthorizationUrl {
				try {
					if (this.requestToken != token) return null;
					return new URL(authService.getAuthorizationUrl(token(Optional.of(token))));
				} catch (Exception exception) {
					throw new CouldNotObtainAuthorizationUrl(exception);
				}
			}

			@Override
			public Token accessToken() {
				return accessToken;
			}

			@SuppressWarnings("unchecked")
			@Override
			public Token accessToken(Verifier verifier) throws CouldNotObtainAccessToken {
				try {
					RestAccessor accessor = new io.intino.alexandria.restaccessor.core.RestAccessor();
					Response response = accessor.post(url(), tokenPath(), tokenRequestParameters(verifier.value()));
					if (response.code() != 200) {
						throw new CouldNotObtainAccessToken(new Exception(response.code() + " in " + tokenUrl() + " " + "Basic " + encodeClientIdAndSecret()));
					}
					Map<String, String> map = new Gson().fromJson(response.content(), Map.class);
					this.accessToken = new Token() {
						@Override
						public String id() {
							return map.get("access_token");
						}

						@Override
						public String secret() {
							return "";
						}
					};
				} catch (Exception e) {
					throw new CouldNotObtainAccessToken(e);
				}

				return this.accessToken;
			}

			@Override
			public void invalidate() throws CouldNotInvalidateAccessToken {
				try {
					api.get(url(), logoutPath(), logoutRequestParameters());
					userInfo = null;
				} catch (Exception exception) {
					throw new CouldNotInvalidateAccessToken(exception);
				}
			}

			@Override
			public Version version() {
				return Version.OAuth2;
			}
		};
	}

	@Override
	public boolean valid(Token token) {
		if (token == null) return false;
		return loadUserInfo(token).containsKey("email");
	}

	@Override
	public FederationInfo info(Token accessToken) {
		return new FederationInfo() {
			@Override
			public String name() {
				return "azure";
			}

			@Override
			public String title() {
				return "Azure federation";
			}

			@Override
			public String subtitle() {
				return null;
			}

			@Override
			public URL logo() {
				return null;
			}

			@Override
			public URI pushServerUri() {
				return null;
			}
		};
	}

	@Override
	public UserInfo me(Token token) throws CouldNotObtainInfo {
		this.userInfo = userInfo(loadUserInfo(token));
		return this.userInfo;
	}

	@Override
	public void logout(Token accessToken) {
	}

	@Override
	public String logoutUrl() {
		return url() + logoutPath();
	}

	@Override
	public void addPushListener(Token accessToken, FederationNotificationListener listener) throws CouldNotObtainInfo {
	}

	private org.scribe.model.Token token(Optional<Token> token) {
		if (!token.isPresent())
			return null;

		return new org.scribe.model.Token(token.get().id(), "");
	}

	private OAuthService authService() throws SpaceAuthCallbackUrlIsNull {
		ServiceBuilder builder = (new ServiceBuilder()).provider(this.apiOf()).apiKey(property(ClientId)).apiSecret(property(ClientSecret));
		URL callbackUrl = callbackUrl(space);
		if (callbackUrl == null) {
			throw new SpaceAuthCallbackUrlIsNull();
		} else {
			builder.callback(callbackUrl.toString());
			return builder.build();
		}
	}

	//private static final String Scope = "2ff814a6-3304-4ab8-85cb-cd0e6f879c1d%2F.default";
	private static final String Scope = "00000003-0000-0000-c000-000000000000%2F.default";
	private static final String CodeVerifier = "YTFjNjI1OWYzMzA3MTI4ZDY2Njg5M2RkNmVjNDE5YmEyZGRhOGYyM2IzNjdmZWFhMTQ1ODg3NDcxY2Nl";
	private static final String AuthorizationUrl = "/%s/oauth2/v2.0/authorize?client_id=%s&response_type=code&redirect_uri=%s&response_mode=query&scope=%s&state=1234&code_challenge=%s";

	private Api apiOf() {
		return new DefaultApi20() {
			public String getAccessTokenEndpoint() {
				return tokenUrl();
			}

			@Override
			public String getAuthorizationUrl(OAuthConfig oAuthConfig) {
				return url() + String.format(AuthorizationUrl, property(TenantId), property(ClientId), callbackUrl(space), Scope, CodeVerifier);
			}
		};
	}

	private String tokenUrl() {
		return url() + tokenPath();
	}

	private static final String TokenPath = "/%s/oauth2/v2.0/token";
	private String tokenPath() {
		return String.format(TokenPath, property(TenantId));
	}

	private Map<String, String> tokenRequestParameters(String code) {
		URL redirectUri = callbackUrl(space);
		return new HashMap<>() {{
			put("code", code);
			put("client_id", property(ClientId));
			put("scope", Scope);
			put("redirect_uri", redirectUri != null ? redirectUri.toString() : "");
			put("grant_type", "authorization_code");
			put("code_verifier", CodeVerifier);
			put("client_secret", property(ClientSecret));
		}};
	}

	private static final String LogoutPath = "/%s/oauth2/logout?client_id=%s&post_logout_redirect_uri=%s";
	private String logoutPath() {
		return String.format(LogoutPath, property(TenantId), property(ClientId), space().url().toString());
	}

	private Map<String, String> logoutRequestParameters() {
		return new HashMap<>() {{
			put("client_id", property(ClientId));
			put("post_logout_redirect_uri", space().url().toString());
		}};
	}

	private static final String RevokePath = "/%s/oauth2/v2.0/token/revoke?token=%s";
	private String revokePath() {
		return String.format(RevokePath, property(TenantId), property(ClientId), space().url().toString());
	}

	private URL callbackUrl(Space space) {
		try {
			return new URL(space().url().toString() + "/authenticate-callback");
		} catch (MalformedURLException e) {
			Logger.error(e);
			return null;
		}
	}

	private UserInfo userInfo(Map<String, Object> map) {
		return new UserInfo() {
			@Override
			public String username() {
				return map.get("email").toString();
			}

			@Override
			public String fullName() {
				return map.get("name").toString();
			}

			@Override
			public URL photo() {
				return null;
			}

			@Override
			public String email() {
				return "";
			}

			@Override
			public String language() {
				return "es";
			}

			@Override
			public List<String> roleList() {
				return Collections.emptyList();
			}
		};
	}

	public UserInfo userInfo() {
		return userInfo;
	}

	public static class HttpClientFactory {

		public static CloseableHttpClient client() throws IOException {
			try {
				SSLContextBuilder builder = new SSLContextBuilder();
				builder.loadTrustMaterial(null, new TrustSelfSignedStrategy());
				SSLConnectionSocketFactory sslConnectionSocketFactory = new SSLConnectionSocketFactory(builder.build(), NoopHostnameVerifier.INSTANCE);
				Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory>create()
						.register("http", new PlainConnectionSocketFactory())
						.register("https", sslConnectionSocketFactory)
						.build();

				PoolingHttpClientConnectionManager cm = new PoolingHttpClientConnectionManager(registry);
				cm.setMaxTotal(100);
				return HttpClients.custom()
						.setSSLSocketFactory(sslConnectionSocketFactory)
						.setConnectionManager(cm)
						.build();
			} catch (NoSuchAlgorithmException | KeyStoreException | KeyManagementException e) {
				throw new IOException("Error getting client");
			}
		}
	}

	private String encodeClientIdAndSecret() {
		return Base64.getEncoder().encodeToString((property(ClientId) + ":" + property(ClientSecret)).getBytes());
	}

	@SuppressWarnings("unchecked")
	private Map<String, Object> loadUserInfo(Token token) {
		HttpGet get = new HttpGet(property(GraphUrl) + "/oidc/userinfo");
		get.setHeader(HttpHeaders.AUTHORIZATION, "Bearer " + token.id());
		try {
			CloseableHttpClient client = HttpClientFactory.client();
			HttpResponse response = client.execute(get);
			if (response.getStatusLine().getStatusCode() != 200) return Collections.emptyMap();
			BufferedReader rd = new BufferedReader(new InputStreamReader(response.getEntity().getContent()));
			StringBuilder result = new StringBuilder();
			String line;
			while ((line = rd.readLine()) != null) result.append(line);
			return new Gson().fromJson(result.toString(), Map.class);
		} catch (IOException e) {
			return Collections.emptyMap();
		}
	}

	private String property(String name) {
		return configuration.property(name);
	}

}
