package io.quassar.archetype;

import java.io.File;
import java.util.Collections;

public class Archetype {
	private final File root;

	public Archetype(File root) {
		this.root = root;
		root.mkdirs();
	}

	public File root() {
		return this.root;
	}

	public Relative relative() {
	    return new Relative(root);
	}

	private static File createParent(File file) {
	    file.getParentFile().mkdirs();
	    return file;
	}

	public Configuration configuration() {
		return new Configuration(root);
	}

	public File index() {
		return createParent(new File(root, "index.triples"));
	}

	public File readme() {
		return createParent(new File(root, "readme.html"));
	}

	public Models models() {
		return new Models(root);
	}

	public Languages languages() {
		return new Languages(root);
	}

	public Projects projects() {
		return new Projects(root);
	}

	public Users users() {
		return new Users(root);
	}

	public Tmp tmp() {
		return new Tmp(root);
	}

	public static class Configuration {
		private final File root;

		public Configuration(File parent) {
			this.root = new File(parent, "configuration");
			root.mkdirs();
		}

		public File root() {
			return this.root;
		}

		public File modelSequence() {
			return createParent(new File(root, "model-sequence.tsv"));
		}

		public File defaultLanguages() {
			return createParent(new File(root, "default-languages.tsv"));
		}

		public File defaultModels() {
			return createParent(new File(root, "default-models.tsv"));
		}

		public Editor editor() {
			return new Editor(root);
		}

		public static class Editor {
			private final File root;

			public Editor(File parent) {
				this.root = new File(parent, "editor");
				root.mkdirs();
			}

			public File root() {
				return this.root;
			}

			public File utilities() {
				return createParent(new File(root, "utilities.txt"));
			}
		}
	}

	public static class Models {
		private final File root;

		public Models(File parent) {
			this.root = new File(parent, "models");
			root.mkdirs();
		}

		public File root() {
			return this.root;
		}

		public File get(String path) {
			return createParent(new File(root, "{path}".replace("{path}", path)));
		}

		public File get(String path, String name) {
			return createParent(new File(root, "{path}/{name}".replace("{path}", path).replace("{name}", name)));
		}

		public File workspace(String path, String name) {
			return createParent(new File(root, "{path}/{name}/workspace".replace("{path}", path).replace("{name}", name)));
		}

		public java.util.List<File> releases(String path, String name) {
		    new File(root, "{path}/{name}/releases".replace("{path}", path).replace("{name}", name)).mkdirs();
			return java.util.Arrays.stream(new File(root, "{path}/{name}/releases".replace("{path}", path).replace("{name}", name)).listFiles()).
		        filter(f-> f.getName().contains(".zip")).
		        collect(java.util.stream.Collectors.toList());
		}

		public File release(String path, String name, String version) {
			return createParent(new File(root, "{path}/{name}/releases/{version}.zip".replace("{path}", path).replace("{name}", name).replace("{version}", version)));
		}
	}

	public static class Languages {
		private final File root;

		public Languages(File parent) {
			this.root = new File(parent, "dsls");
			root.mkdirs();
		}

		public File root() {
			return this.root;
		}

		public File get(String dsl) {
			return createParent(new File(root, "{dsl}".replace("{dsl}", dsl)));
		}

		public File logo(String dsl) {
			return createParent(new File(root, "{dsl}/logo.png".replace("{dsl}", dsl)));
		}

		public File releases(String dsl) {
			return createParent(new File(root, "{dsl}/releases".replace("{dsl}", dsl)));
		}

		public File release(String dsl, String version) {
			return createParent(new File(root, "{dsl}/releases/{version}".replace("{dsl}", dsl).replace("{version}", version)));
		}

		public File releaseDslJar(String dsl, String version) {
			return createParent(new File(root, "{dsl}/releases/{version}/dsl.jar".replace("{dsl}", dsl).replace("{version}", version)));
		}

		public File releaseDslManifest(String dsl, String version) {
			return createParent(new File(root, "{dsl}/releases/{version}/dsl.xml".replace("{dsl}", dsl).replace("{version}", version)));
		}

		public File releaseHelp(String dsl, String version) {
			return createParent(new File(root, "{dsl}/releases/{version}/help.html".replace("{dsl}", dsl).replace("{version}", version)));
		}

		public File releaseGraph(String dsl, String version) {
			return createParent(new File(root, "{dsl}/releases/{version}/graph.json".replace("{dsl}", dsl).replace("{version}", version)));
		}

		public java.util.List<File> releaseParsers(String dsl, String version) {
		    new File(root, "{dsl}/releases/{version}/parsers".replace("{dsl}", dsl).replace("{version}", version)).mkdirs();
			return java.util.Arrays.stream(new File(root, "{dsl}/releases/{version}/parsers".replace("{dsl}", dsl).replace("{version}", version)).listFiles()).
		        filter(f-> f.getName().contains(".zip")).
		        collect(java.util.stream.Collectors.toList());
		}

		public File releaseParsersDir(String dsl, String version) {
			return createParent(new File(root, "{dsl}/releases/{version}/parsers".replace("{dsl}", dsl).replace("{version}", version)));
		}

		public File releaseParserDir(String dsl, String version, String name) {
			return createParent(new File(root, "{dsl}/releases/{version}/parsers/{name}".replace("{dsl}", dsl).replace("{version}", version).replace("{name}", name)));
		}

		public File releaseParserFile(String dsl, String version, String name) {
			return createParent(new File(root, "{dsl}/releases/{version}/parsers/{name}.zip".replace("{dsl}", dsl).replace("{version}", version).replace("{name}", name)));
		}

		public File releaseParserJar(String dsl, String version, String name) {
			return createParent(new File(root, "{dsl}/releases/{version}/parsers/{name}.jar".replace("{dsl}", dsl).replace("{version}", version).replace("{name}", name)));
		}

		public File releaseParserManifest(String dsl, String version, String name) {
			return createParent(new File(root, "{dsl}/releases/{version}/parsers/{name}.xml".replace("{dsl}", dsl).replace("{version}", version).replace("{name}", name)));
		}
	}

	public static class Projects {
		private final File root;

		public Projects(File parent) {
			this.root = new File(parent, "projects");
			root.mkdirs();
		}

		public File root() {
			return this.root;
		}
	}

	public static class Users {
		private final File root;

		public Users(File parent) {
			this.root = new File(parent, "users");
			root.mkdirs();
		}

		public File root() {
			return this.root;
		}

		public File user(String username) {
			return createParent(new File(root, "{username}".replace("{username}", username)));
		}

		public File photo(String username) {
			return createParent(new File(root, "{username}/photo.png".replace("{username}", username)));
		}
	}

	public static class Tmp {
		private final File root;

		public Tmp(File parent) {
			this.root = new File(parent, "tmp");
			root.mkdirs();
		}

		public File root() {
			return this.root;
		}

		public File upload(String name) {
			return createParent(new File(root, "uploads/{name}".replace("{name}", name)));
		}

		public File builds(String name) {
			return createParent(new File(root, "builds/{name}".replace("{name}", name)));
		}

		public File model(String language, String model) {
			return createParent(new File(root, "{language}/{model}".replace("{language}", language).replace("{model}", model)));
		}

		public File releaseWorkspace(String model, String version) {
			return createParent(new File(root, "{model}/{version}".replace("{model}", model).replace("{version}", version)));
		}
	}

	public static class Relative {
        private final String name;
        private final File root;

        public Relative(File root) {
            this.name = "";
            this.root = root;
            root.mkdirs();
        }

        public String root() {
            return this.name;
        }

        public Configuration configuration() {
        	return new Configuration(root, name);
        }

        public String index() {
        	return "index.triples";
        }

        public String readme() {
        	return "readme.html";
        }

        public Models models() {
        	return new Models(root, name);
        }

        public Languages languages() {
        	return new Languages(root, name);
        }

        public Projects projects() {
        	return new Projects(root, name);
        }

        public Users users() {
        	return new Users(root, name);
        }

        public Tmp tmp() {
        	return new Tmp(root, name);
        }

        public static class Configuration {
        	private final File root;
        	private final String name;

        	public Configuration(File parentRoot, String parent) {
        		this.root = new File(parent, "configuration");
        		root.mkdirs();
        		this.name = parent + (!parent.isEmpty() ? "/" : "") + "configuration";
        	}

        	public String root() {
        		return this.name;
        	}

        	public String modelSequence() {
        		return "model-sequence.tsv";
        	}

        	public String defaultLanguages() {
        		return "default-languages.tsv";
        	}

        	public String defaultModels() {
        		return "default-models.tsv";
        	}

        	public Editor editor() {
        		return new Editor(root, name);
        	}

        	public static class Editor {
        		private final File root;
        		private final String name;

        		public Editor(File parentRoot, String parent) {
        			this.root = new File(parent, "editor");
        			root.mkdirs();
        			this.name = parent + (!parent.isEmpty() ? "/" : "") + "editor";
        		}

        		public String root() {
        			return this.name;
        		}

        		public String utilities() {
        			return "utilities.txt";
        		}
        	}
        }

        public static class Models {
        	private final File root;
        	private final String name;

        	public Models(File parentRoot, String parent) {
        		this.root = new File(parent, "models");
        		root.mkdirs();
        		this.name = parent + (!parent.isEmpty() ? "/" : "") + "models";
        	}

        	public String root() {
        		return this.name;
        	}

        	public String get(String path) {
        		return "{path}".replace("{path}", path);
        	}

        	public String get(String path, String name) {
        		return "{path}/{name}".replace("{path}", path).replace("{name}", name);
        	}

        	public String workspace(String path, String name) {
        		return "{path}/{name}/workspace".replace("{path}", path).replace("{name}", name);
        	}

        	public java.util.List<String> releases(String path, String name) {
        	    if(!new File(root, "{path}/{name}/releases").exists()) return Collections.emptyList();
        		return java.util.Arrays.stream(new File(root, "{path}/{name}/releases").listFiles()).
        	        filter(f-> f.getName().contains(".zip".replace("{path}", path).replace("{name}", name))).
        	        map(f -> f.getName()).
        	        collect(java.util.stream.Collectors.toList());
        	}

        	public String release(String path, String name, String version) {
        		return "{path}/{name}/releases/{version}.zip".replace("{path}", path).replace("{name}", name).replace("{version}", version);
        	}
        }

        public static class Languages {
        	private final File root;
        	private final String name;

        	public Languages(File parentRoot, String parent) {
        		this.root = new File(parent, "dsls");
        		root.mkdirs();
        		this.name = parent + (!parent.isEmpty() ? "/" : "") + "dsls";
        	}

        	public String root() {
        		return this.name;
        	}

        	public String get(String dsl) {
        		return "{dsl}".replace("{dsl}", dsl);
        	}

        	public String logo(String dsl) {
        		return "{dsl}/logo.png".replace("{dsl}", dsl);
        	}

        	public String releases(String dsl) {
        		return "{dsl}/releases".replace("{dsl}", dsl);
        	}

        	public String release(String dsl, String version) {
        		return "{dsl}/releases/{version}".replace("{dsl}", dsl).replace("{version}", version);
        	}

        	public String releaseDslJar(String dsl, String version) {
        		return "{dsl}/releases/{version}/dsl.jar".replace("{dsl}", dsl).replace("{version}", version);
        	}

        	public String releaseDslManifest(String dsl, String version) {
        		return "{dsl}/releases/{version}/dsl.xml".replace("{dsl}", dsl).replace("{version}", version);
        	}

        	public String releaseHelp(String dsl, String version) {
        		return "{dsl}/releases/{version}/help.html".replace("{dsl}", dsl).replace("{version}", version);
        	}

        	public String releaseGraph(String dsl, String version) {
        		return "{dsl}/releases/{version}/graph.json".replace("{dsl}", dsl).replace("{version}", version);
        	}

        	public java.util.List<String> releaseParsers(String dsl, String version) {
        	    if(!new File(root, "{dsl}/releases/{version}/parsers").exists()) return Collections.emptyList();
        		return java.util.Arrays.stream(new File(root, "{dsl}/releases/{version}/parsers").listFiles()).
        	        filter(f-> f.getName().contains(".zip".replace("{dsl}", dsl).replace("{version}", version))).
        	        map(f -> f.getName()).
        	        collect(java.util.stream.Collectors.toList());
        	}

        	public String releaseParsersDir(String dsl, String version) {
        		return "{dsl}/releases/{version}/parsers".replace("{dsl}", dsl).replace("{version}", version);
        	}

        	public String releaseParserDir(String dsl, String version, String name) {
        		return "{dsl}/releases/{version}/parsers/{name}".replace("{dsl}", dsl).replace("{version}", version).replace("{name}", name);
        	}

        	public String releaseParserFile(String dsl, String version, String name) {
        		return "{dsl}/releases/{version}/parsers/{name}.zip".replace("{dsl}", dsl).replace("{version}", version).replace("{name}", name);
        	}

        	public String releaseParserJar(String dsl, String version, String name) {
        		return "{dsl}/releases/{version}/parsers/{name}.jar".replace("{dsl}", dsl).replace("{version}", version).replace("{name}", name);
        	}

        	public String releaseParserManifest(String dsl, String version, String name) {
        		return "{dsl}/releases/{version}/parsers/{name}.xml".replace("{dsl}", dsl).replace("{version}", version).replace("{name}", name);
        	}
        }

        public static class Projects {
        	private final File root;
        	private final String name;

        	public Projects(File parentRoot, String parent) {
        		this.root = new File(parent, "projects");
        		root.mkdirs();
        		this.name = parent + (!parent.isEmpty() ? "/" : "") + "projects";
        	}

        	public String root() {
        		return this.name;
        	}
        }

        public static class Users {
        	private final File root;
        	private final String name;

        	public Users(File parentRoot, String parent) {
        		this.root = new File(parent, "users");
        		root.mkdirs();
        		this.name = parent + (!parent.isEmpty() ? "/" : "") + "users";
        	}

        	public String root() {
        		return this.name;
        	}

        	public String user(String username) {
        		return "{username}".replace("{username}", username);
        	}

        	public String photo(String username) {
        		return "{username}/photo.png".replace("{username}", username);
        	}
        }

        public static class Tmp {
        	private final File root;
        	private final String name;

        	public Tmp(File parentRoot, String parent) {
        		this.root = new File(parent, "tmp");
        		root.mkdirs();
        		this.name = parent + (!parent.isEmpty() ? "/" : "") + "tmp";
        	}

        	public String root() {
        		return this.name;
        	}

        	public String upload(String name) {
        		return "uploads/{name}".replace("{name}", name);
        	}

        	public String builds(String name) {
        		return "builds/{name}".replace("{name}", name);
        	}

        	public String model(String language, String model) {
        		return "{language}/{model}".replace("{language}", language).replace("{model}", model);
        	}

        	public String releaseWorkspace(String model, String version) {
        		return "{model}/{version}".replace("{model}", model).replace("{version}", version);
        	}
        }
	}
}