package io.intino.tara.processors.parser;

import io.intino.tara.Source;
import io.intino.tara.language.grammar.SyntaxException;
import io.intino.tara.language.grammar.TaraGrammar;
import io.intino.tara.language.grammar.TaraLexer;
import io.intino.tara.processors.model.Model;
import io.intino.tara.processors.parser.antlr.ConstraintsModelGenerator;
import io.intino.tara.processors.parser.antlr.ModelGenerator;
import io.intino.tara.processors.parser.antlr.TaraErrorStrategy;
import org.antlr.v4.runtime.CommonTokenStream;
import org.antlr.v4.runtime.DefaultErrorStrategy;
import org.antlr.v4.runtime.RecognitionException;
import org.antlr.v4.runtime.Token;
import org.antlr.v4.runtime.tree.ParseTreeWalker;

import java.io.IOException;
import java.io.InputStream;
import java.util.logging.Level;
import java.util.logging.Logger;

import static org.antlr.v4.runtime.CharStreams.fromString;

public class Parser {
	private static final Logger LOG = Logger.getGlobal();
	private final Source source;
	private final DefaultErrorStrategy errorHandler;

	public Parser(Source source, DefaultErrorStrategy errorStrategy) {
		this.source = source;
		this.errorHandler = errorStrategy;
	}

	public Parser(Source source) {
		this(source, new TaraErrorStrategy());
	}

	public TaraGrammar.RootContext parse() throws SyntaxException {
		return parse(false);
	}

	public TaraGrammar.RootContext parse(boolean trace) throws SyntaxException {
		try (InputStream content = source.content()) {
			TaraLexer lexer = new TaraLexer(fromString(new String(content.readAllBytes()), source.charset().name()));
			lexer.reset();
			CommonTokenStream tokens = new CommonTokenStream(lexer);
			TaraGrammar grammar = new TaraGrammar(tokens);
			grammar.setErrorHandler(errorHandler);
			grammar.setTrace(trace);
			grammar.addErrorListener(new GrammarErrorListener());
			return grammar.root();
		} catch (RecognitionException e) {
			LOG.log(Level.INFO, e.getMessage(), e);
			throwError(e);
			return null;
		} catch (IOException e) {
			LOG.log(Level.SEVERE, e.getMessage());
			return null;
		}
	}

	public Model convert(TaraGrammar.RootContext rootContext) throws SyntaxException {
		try {
			ParseTreeWalker walker = new ParseTreeWalker();
			ModelGenerator extractor = new ModelGenerator(source);
			walker.walk(extractor, rootContext);
			if (!extractor.getErrors().isEmpty()) throw extractor.getErrors().get(0);
			Model model = extractor.getModel();
			ConstraintsModelGenerator constraintsGenerator = new ConstraintsModelGenerator(source, rootContext);
			model.constraints(constraintsGenerator.walk());
			if (!constraintsGenerator.errors().isEmpty()) throw constraintsGenerator.errors().get(0);
			return model;
		} catch (RecognitionException e) {
			LOG.log(Level.SEVERE, e.getMessage());
			return throwError(e);
		}
	}

	private Model throwError(RecognitionException e) throws SyntaxException {
		org.antlr.v4.runtime.Parser recognizer = (org.antlr.v4.runtime.Parser) e.getRecognizer();
		Token token = recognizer.getCurrentToken();
		throw new SyntaxException("Syntax error in " + source.uri(), source.uri(), token.getLine(), token.getCharPositionInLine(), getExpectedTokens(recognizer));
	}

	private String getExpectedTokens(org.antlr.v4.runtime.Parser recognizer) {
		try {
			return recognizer.getExpectedTokens().toString(recognizer.getVocabulary());
		} catch (Exception e) {
			LOG.log(Level.INFO, e.getMessage(), e);
			return "";
		}
	}
}