package io.intino.tara;

import io.intino.tara.language.model.Element;
import io.intino.tara.language.model.Mogram;
import io.intino.tara.language.semantics.Assumption;
import io.intino.tara.language.semantics.Constraint;
import io.intino.tara.language.semantics.errorcollector.SemanticException;
import io.intino.tara.language.semantics.errorcollector.SemanticFatalException;
import io.intino.tara.language.semantics.errorcollector.SemanticNotification;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import static io.intino.tara.language.semantics.constraints.FacetConstraint.findFacet;
import static io.intino.tara.language.semantics.errorcollector.SemanticNotification.Level.ERROR;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;

public class Checker {

	private final Resolver resolver;
	private final Language language;
	private final List<SemanticException> exceptions = new ArrayList<>();

	public Checker(Language language) {
		this.language = language;
		this.resolver = new Resolver(language);
	}

	public void check(Mogram mogram) throws SemanticFatalException {
		exceptions.clear();
		resolver.resolve(mogram);
		checkConstraints(mogram);
		if (!exceptions.isEmpty()) {
			recoverErrors();
			if (!exceptions.isEmpty()) throw new SemanticFatalException(exceptions);
		}
	}

	private void recoverErrors() {
		final List<SemanticException> toRemove = exceptions.stream().filter(e -> e.getNotification().key().equals("required.parameter.in.context") && isParameterNotFoundRecoverable(e.getNotification().origin()[0], e.getNotification().parameters().get(0).toString(), e.getNotification().parameters().get(1).toString())).collect(toList());
		exceptions.removeAll(toRemove);
	}

	private boolean isParameterNotFoundRecoverable(Element element, String name, String type) {
		final Mogram mogram = (Mogram) element;
		if (language == null || mogram == null || mogram.type() == null) return false;
		final List<Constraint.Facet> facets = language.constraints(mogram.type()).stream().filter(c -> sameFacet(mogram, c)).map(c -> (Constraint.Facet) c).collect(toList());
		for (Constraint.Facet facet : facets)
			for (Constraint.Parameter c : facet.constraints().stream().filter(c -> c instanceof Constraint.Parameter).map(p -> (Constraint.Parameter) p).collect(toList()))
				if (c.type().name().equalsIgnoreCase(type) && c.name().equals(name) && !c.size().isRequired()) return true;
		return false;
	}

	private boolean sameFacet(Mogram mogram, Constraint c) {
		return c instanceof Constraint.Facet && findFacet(mogram, ((Constraint.Facet) c).type()) != null;
	}

	private void checkConstraints(Mogram mogram) throws SemanticFatalException {
		if (mogram == null)
			throw new SemanticFatalException(new SemanticNotification(SemanticNotification.Level.ERROR, "Node is null", (Element) null));
		assume(mogram);
		checkNodeConstrains(mogram);
	}

	private void assume(Mogram mogram) {
		if (mogram == null || mogram.type() == null || language == null) return;
		List<Assumption> assumptions = language.assumptions(mogram.type());
		if (assumptions != null) assume(mogram, assumptions);
		for (String type : mogram.secondaryTypes()) {
			assumptions = language.assumptions(type);
			if (assumptions != null) assume(mogram, assumptions);
		}
	}

	private void assume(Mogram mogram, Collection<Assumption> assumptions) {
		for (Assumption assumption : assumptions)
			assumption.assume(mogram);
	}

	private void checkNodeConstrains(Mogram mogram) throws SemanticFatalException {
		Collection<Constraint> constraints = language.constraints(mogram.type());
		if (constraints == null) finish(mogram);
		else for (Constraint constraint : constraints)
			try {
				constraint.check(mogram);
			} catch (SemanticException e) {
				if (e.level() == ERROR && e.isFatal()) throw new SemanticFatalException(singletonList(e));
				else exceptions.add(e);
			}
	}

	private void finish(Mogram mogram) throws SemanticFatalException {
		if (!mogram.isReference())
			throw new SemanticFatalException(new SemanticNotification(ERROR, "reject.type.not.exists", mogram, singletonList(presentableType(mogram.type()))));
	}

	private String presentableType(String type) {
		return type.replaceFirst(":", " on ");
	}
}