package io.intino.tara.language.semantics.constraints.flags;

import io.intino.tara.dsls.MetaIdentifiers;
import io.intino.tara.language.model.Mogram;
import io.intino.tara.language.model.MogramRoot;
import io.intino.tara.language.model.Tag;
import io.intino.tara.language.model.rules.Size;
import io.intino.tara.language.semantics.errorcollector.SemanticException;
import io.intino.tara.language.semantics.errorcollector.SemanticNotification;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static io.intino.tara.language.model.Tag.*;
import static io.intino.tara.language.semantics.errorcollector.SemanticNotification.Level.ERROR;
import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;

public class FlagCoherenceCheckerFactory {

	private static final Map<String, FlagChecker> checkers = new HashMap<>();

	static {
		checkers.put(Tag.Private.name().toLowerCase(), new PrivateChecker());
		checkers.put(Feature.name().toLowerCase(), new FeatureChecker());
		checkers.put(Tag.Component.name().toLowerCase(), new ComponentChecker());
	}

	private FlagCoherenceCheckerFactory() {
	}


	public static FlagChecker get(Object key) {
		return checkers.get(key.toString());
	}

	private static class PrivateChecker implements FlagChecker {
		@Override
		public void check(Mogram mogram) throws SemanticException {
			if (mogram.flags().contains(Final)) throw error(mogram, asList(Final.name(), Private.name()));
		}
	}

	private static class FeatureChecker implements FlagChecker {
		@Override
		public void check(Mogram mogram) throws SemanticException {
			if (mogram.type().equals(MetaIdentifiers.META_CONCEPT)) throw error("metaconcept.cannot.be", mogram, singletonList(Feature.name()));
			if (mogram.isReference() && !mogram.targetOfReference().is(Feature))
				throw error("declared.node.must.be", mogram, singletonList(Feature.name()));
		}
	}

	private static class ComponentChecker implements FlagChecker {
		@Override
		public void check(Mogram mogram) throws SemanticException {
			if (mogram.isReference() || !(mogram.container() instanceof MogramRoot)) return;
			if (mogram.isReference() && !mogram.targetOfReference().is(Component))
				throw error("declared.node.must.be", mogram, singletonList(Component.name()));
			if (mogram.flags().contains(Feature)) throw error(mogram, asList(Feature.name(), Component.name()));
			final Size size = mogram.container().sizeOf(mogram);
			if (size == null) return;
			if (size.min() != 0 || size.max() != Integer.MAX_VALUE) throw error("reject.root.component.size", mogram);
		}
	}

	public static SemanticException error(Mogram mogram, List<String> flags) {
		return new SemanticException(new SemanticNotification(ERROR, "reject.flag.combination", mogram, flags));
	}

	public static SemanticException error(String message, Mogram mogram) {
		return new SemanticException(new SemanticNotification(ERROR, message, mogram));
	}

	public static SemanticException error(String message, Mogram mogram, List<String> parameters) {
		return new SemanticException(new SemanticNotification(ERROR, message, mogram, parameters));
	}
}
