package io.intino.tara.language.semantics.constraints.component;

import io.intino.tara.Resolver;
import io.intino.tara.language.model.*;
import io.intino.tara.language.model.rules.MogramRule;
import io.intino.tara.language.model.rules.Size;
import io.intino.tara.language.semantics.Constraint;
import io.intino.tara.language.semantics.errorcollector.SemanticException;
import io.intino.tara.language.semantics.errorcollector.SemanticNotification;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

public class Component implements Constraint.Component {
	private final String type;
	private final List<Rule> rules;
	private final List<Tag> annotations;

	public Component(String type, List<Rule> rules, List<Tag> annotations) {
		this.type = type;
		this.rules = rules;
		this.annotations = annotations;
	}

	@Override
	public String type() {
		return type;
	}

	public MogramRule compositionRule() {
		return (MogramRule) rules.stream().filter(r -> r instanceof MogramRule).findFirst().orElse(null);
	}

	public List<Rule> rules() {
		return rules;
	}

	@Override
	public List<Tag> annotations() {
		return annotations;
	}

	@Override
	public void check(Element element) throws SemanticException {
		Mogram container = (Mogram) element;
		if (container.isReference()) return;
		List<Mogram> components = filterByType(container);
		final List<Mogram> accepted = acceptedComponents(components);
		if (!accepted.isEmpty()) components.forEach(this::addFlags);
		final List<Mogram> notAccepted = notAccepted(components, accepted);
		if (!notAccepted.isEmpty()) error(notAccepted.get(0));
		else checkRequired(element, accepted);
	}

	private List<Mogram> acceptedComponents(List<Mogram> components) {
		return components.stream().filter(component -> rules.stream().allMatch(r -> accept(r, components, component))).collect(Collectors.toList());
	}

	private boolean accept(Rule r, List<Mogram> components, Mogram component) {
		return r instanceof Size ? r.accept(components) : r.accept(component);
	}

	private List<Mogram> notAccepted(List<Mogram> components, List<Mogram> accepted) {
		return components.stream().filter(c -> !accepted.contains(c)).collect(Collectors.toList());
	}

	private void error(Mogram notAccepted) throws SemanticException {
		for (Rule rule : rules)
			if (!accept(rule, notAccepted.container().components(), notAccepted))
				throw new SemanticException(new SemanticNotification(SemanticNotification.Level.ERROR, rule.errorMessage(), notAccepted, rule.errorParameters()));
	}

	private void checkRequired(Element element, List<Mogram> accepted) throws SemanticException {
		if (rules.get(0) instanceof Size && ((Size) rules.get(0)).isRequired() && !isAccepted(accepted, type())) {
			String message = "required.type.in.context";
			List<?> parameters = Collections.singletonList(this.type.replace(":", " on "));
			throw new SemanticException(new SemanticNotification(SemanticNotification.Level.ERROR, message, element, parameters));
		}
	}

	private boolean isAccepted(List<Mogram> accepted, String type) {
		for (Mogram mogram : accepted) if (mogram.type().equals(type)) return true;
		return false;
	}

	private void addFlags(Mogram mogram) {
		List<Tag> flags = new ArrayList<>(mogram.flags());
		for (Tag flag : this.annotations) {
			if (!flags.contains(flag)) mogram.addFlags(flag);
			flags.add(flag);
		}
	}

	private List<Mogram> filterByType(MogramContainer node) {
		return node.components().stream().filter(this::isCompatibles).collect(Collectors.toList());
	}

	private boolean isCompatibles(Mogram mogram) {
		for (String nodeType : mogram.types())
			if (nodeType != null && (nodeType.equals(type) || nodeType.equals(Resolver.shortType(type)))) return true;
		return checkFacets(mogram);
	}

	private boolean checkFacets(Mogram mogram) {
		for (io.intino.tara.language.model.Facet facet : mogram.appliedFacets())
			if (facet.type().equals(Resolver.shortType(type))) return true;
		return false;
	}

	@Override
	public String toString() {
		return "Component{" + type + '}';
	}
}
