package io.intino.tara.language.semantics.constraints;

import io.intino.tara.Resolver;
import io.intino.tara.language.model.*;
import io.intino.tara.language.model.rules.Size;
import io.intino.tara.language.model.rules.variable.VariableRule;
import io.intino.tara.language.semantics.Assumption;
import io.intino.tara.language.semantics.Assumption.FacetInstance;
import io.intino.tara.language.semantics.Constraint;
import io.intino.tara.language.semantics.constraints.component.Component;
import io.intino.tara.language.semantics.constraints.component.OneOf;
import io.intino.tara.language.semantics.constraints.parameter.PrimitiveParameter;
import io.intino.tara.language.semantics.constraints.parameter.ReferenceParameter;
import io.intino.tara.language.semantics.errorcollector.SemanticException;
import io.intino.tara.language.semantics.errorcollector.SemanticNotification;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import static io.intino.tara.language.model.Tag.FacetInstance;
import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;

public class RuleFactory {
	private RuleFactory() {
	}

	public static Constraint.Component component(final String type, List<Rule> rules, final Tag... flags) {
		return new Component(type, rules, asList(flags));
	}

	@Deprecated
	public static Constraint.Component component(final String type, Rule rule, final Tag... flags) {
		return new Component(type, singletonList(rule), asList(flags));
	}

	public static Constraint.OneOf oneOf(List<Rule> rules, final Constraint.Component... components) {
		return new OneOf(asList(components), rules);
	}

	public static Constraint.Parameter parameter(final String name, final Primitive type, String facet, final Size size, final int position, String scope, VariableRule rule, Tag... tags) {
		return new PrimitiveParameter(name, type, facet, size, position, scope, rule, asList(tags));
	}

	public static Constraint.Parameter parameter(final String name, String type, String facet, final Size size, final int position, String scope, VariableRule rule, Tag... tags) {
		return new ReferenceParameter(name, type, facet, size, position, scope, rule, asList(tags));
	}

	public static Constraint.Facet facet(final String type, boolean terminal, String[] with, String[] without) {
		return facet(type, terminal, false, with, without);
	}

	public static Constraint.Facet facet(final String type, boolean terminal, boolean required, String[] with, String[] without) {
		return new FacetConstraint(type, terminal, required, with, without);
	}

	public static Constraint.MetaFacet metaFacet(final String type, String... with) {
		return new MetaFacetConstraint(type, with);
	}

	public static Constraint.ComponentNotFound rejectOtherComponents(List<String> types) {
		return new Constraint.ComponentNotFound() {

			@Override
			public void check(Element element) throws SemanticException {
				MogramContainer node = (MogramContainer) element;
				for (Mogram component : node.components()) {
					if (!areCompatibles(component, types))
						throw new SemanticException(new SemanticNotification(SemanticNotification.Level.ERROR, "reject.type.not.exists", component, singletonList(component.type().replace(":", ""))));
				}
			}
		};
	}

	private static boolean areCompatibles(Mogram mogram, List<String> allowedTypes) {
		return mogram.types().stream().anyMatch(type -> type != null &&
				(allowedTypes.contains(type) || (mogram.container() != null && fromFacet(mogram.container().appliedFacets(), type, allowedTypes))))
				|| checkFacet(mogram, allowedTypes);
	}

	public static Constraint.RejectOtherParameters rejectOtherParameters(List<Constraint.Parameter> parameters) {
		return new Constraint.RejectOtherParameters() {
			@Override
			public void check(Element element) throws SemanticException {
				Parametrized parametrized = (Parametrized) element;
				for (io.intino.tara.language.model.Parameter parameter : parametrized.parameters())
					if (!isAcceptable(parameter, parameters))
						throw new SemanticException(new SemanticNotification(SemanticNotification.Level.ERROR, "reject.other.parameter.in.context", parameter, singletonList(parameter.name())));
			}

			private boolean isAcceptable(io.intino.tara.language.model.Parameter parameter, List<Parameter> parameters) {
				return parameters.stream()
						.anyMatch(constraint -> constraint.name().equals(parameter.name()) && hasFacet(constraint.facet(), parameter.container().appliedFacets()));
			}

			private boolean hasFacet(String requiredFacet, List<io.intino.tara.language.model.Facet> facets) {
				return requiredFacet.isEmpty() || facets.stream().anyMatch(facet -> facet.type().equals(requiredFacet));
			}
		};
	}

	public static Constraint.RejectOtherFacets rejectOtherFacets(List<Constraint.Facet> facets) {
		return new Constraint.RejectOtherFacets() {
			@Override
			public void check(Element element) throws SemanticException {
				Mogram mogram = (Mogram) element;
				for (io.intino.tara.language.model.Facet facet : mogram.appliedFacets()) {
					if (!isAcceptable(facets, facet))
						throw new SemanticException(new SemanticNotification(SemanticNotification.Level.ERROR, "reject.other.facet.in.context", facet, singletonList(facet.type())));
				}
			}

			private boolean isAcceptable(List<Facet> facets, io.intino.tara.language.model.Facet facet) {
				return facets.stream().anyMatch(a -> a.type().equals(facet.fullType()));
			}
		};
	}

	private static boolean fromFacet(List<Facet> facets, String nodeType, List<String> types) {
		return facetComponent(facets, nodeType, types) || asFacet(facets, nodeType.split(":")[0]);
	}

	private static boolean facetComponent(List<Facet> facets, String nodeType, List<String> types) {
		return facets.stream().anyMatch(facet -> types.contains(nodeType));
	}

	private static boolean asFacet(List<Facet> facets, String facet) {
		return facets.stream().anyMatch(a -> a.type().equals(facet));
	}

	private static boolean checkFacet(Mogram mogram, List<String> types) {
		List<String> shortTypes = types.stream().map(Resolver::shortType).collect(Collectors.toList());
		return mogram.appliedFacets().stream().anyMatch(facet -> shortTypes.contains(facet.type()));
	}

	public static Constraint name() {
		return new Constraint.Name() {
			@Override
			public void check(Element element) throws SemanticException {
				Mogram mogram = (Mogram) element;
				if (!mogram.isReference() && mogram.name().isEmpty())
					throw new SemanticException(new SemanticNotification(SemanticNotification.Level.ERROR, "required.name", element, Collections.emptyList()));
			}
		};
	}

	public static Constraint.TerminalVariableRedefinition redefine(final String name, String superType) {
		return new Constraint.TerminalVariableRedefinition() {
			@Override
			public void check(Element element) throws SemanticException {
				Mogram mogram = (Mogram) element;
				if (!mogram.flags().contains(Tag.Instance)) {
					for (Variable variable : mogram.variables())
						if (name.equals(variable.name())) return;
					throw new SemanticException(new SemanticNotification(SemanticNotification.Level.ERROR, "required.terminal.variable.redefine", mogram, asList(name, superType)));
				}
			}
		};
	}

	public static Assumption isFacet() {
		return new Assumption.Facet() {
			@Override
			public void assume(Mogram mogram) {
				if (!mogram.flags().contains(Tag.Facet)) mogram.addFlags(Tag.Facet);
				if (!mogram.flags().contains(Tag.Terminal)) mogram.addFlags(Tag.Terminal);
			}
		};
	}

	public static Assumption isFacetInstance() {
		return new FacetInstance() {
			@Override
			public void assume(Mogram mogram) {
				if (!mogram.flags().contains(FacetInstance)) mogram.addFlags(FacetInstance);
			}
		};
	}

	public static Assumption isFeature() {
		return new Assumption.Feature() {
			@Override
			public void assume(Mogram mogram) {
				if (!mogram.flags().contains(Tag.Feature)) mogram.addFlags(Tag.Feature);
				propagateFlags(mogram, Tag.Feature);
			}
		};
	}

	public static Assumption isTerminal() {
		return new Assumption.Terminal() {
			@Override
			public void assume(Mogram mogram) {
				if (mogram.isReference()) return;
				if (!mogram.flags().contains(Tag.Terminal)) mogram.addFlags(Tag.Terminal);
				mogram.variables().stream().filter(variable -> !variable.flags().contains(Tag.Terminal)).forEach(variable -> variable.addFlags(Tag.Terminal));
				propagateFlags(mogram, Tag.Terminal);
			}
		};
	}

	public static Assumption isVolatile() {
		return new Assumption.Volatile() {
			@Override
			public void assume(Mogram mogram) {
				if (mogram.isReference()) return;
				if (!mogram.flags().contains(Tag.Volatile)) mogram.addFlags(Tag.Volatile);
				propagateFlags(mogram, Tag.Volatile);
			}
		};
	}

	public static Assumption isComponent() {
		return new Assumption.Component() {
			@Override
			public void assume(Mogram mogram) {
				if (!mogram.flags().contains(Tag.Component)) mogram.addFlags(Tag.Component);
			}
		};
	}


	public static Assumption isInstance() {
		return new Assumption.Instance() {
			@Override
			public void assume(Mogram mogram) {
				if (!mogram.flags().contains(Tag.Instance)) mogram.addFlags(Tag.Instance);
				mogram.variables().stream().filter(variable -> !variable.flags().contains(Tag.Instance)).forEach(variable -> variable.addFlags(Tag.Instance));
				propagateFlags(mogram, Tag.Instance);
			}
		};
	}

	public static Assumption stashNodeName(String stashNodeName) {
		return new Assumption.StashNodeName() {
			@Override
			public String stashNodeName() {
				return stashNodeName;
			}

			@Override
			public void assume(Mogram mogram) {
				mogram.stashNodeName(stashNodeName);
			}
		};
	}

	private static void propagateFlags(Mogram mogram, Tag tag) {
		for (Mogram component : mogram.components()) {
			if (component.isReference()) continue;
			if (!component.flags().contains(tag)) {
				component.addFlags(tag);
				if (!component.equals(mogram)) propagateFlags(component, tag);
			}
		}
	}

}
