package io.intino.tara;

import io.intino.tara.language.model.Facet;
import io.intino.tara.language.model.Mogram;
import io.intino.tara.language.model.Parameter;
import io.intino.tara.language.semantics.Constraint;

import java.util.Collections;
import java.util.List;

import static java.util.stream.Collectors.toList;

public class Resolver {
	private final Language language;

	public Resolver(Language language) {
		this.language = language;
	}

	public static String shortType(String absoluteType) {
		return absoluteType.contains(".") ? absoluteType.substring(absoluteType.lastIndexOf('.') + 1) : absoluteType;
	}

	public void resolve(Mogram mogram) {
		if (context(mogram) == null) return;
		resolveNode(mogram);
	}

	private void resolveNode(Mogram mogram) {
		resolve(context(mogram));
		List<Constraint> contextConstraints = contextConstraints(mogram);
		if (contextConstraints == null) return;
		for (Constraint constraint : components(contextConstraints))
			if (checkComponentConstraint(mogram, constraint)) return;
		metaFacets(contextConstraints).forEach(constraint -> checkMetaFacetConstraint(mogram, constraint));
	}

	private List<Constraint> contextConstraints(Mogram mogram) {
		if (mogram == null || language == null) return Collections.emptyList();
		final Mogram context = context(mogram);
		List<Constraint> constraints = context != null && context.type() != null ? language.constraints(context.type()) : null;
		if (constraints != null && (isComponent(constraints, mogram) || isMetaFacet(constraints, mogram)))
			return constraints;
		return findInFacets(mogram);
	}

	private boolean isMetaFacet(List<Constraint> constraints, Mogram mogram) {
		return metaFacets(constraints).stream().anyMatch(constraint -> shortType(constraint.type()).equals(mogram.type()));
	}

	private boolean isComponent(List<Constraint> context, Mogram mogram) {
		return context.stream().anyMatch(constraint -> constraint instanceof Constraint.Component &&
				(shortType(((Constraint.Component) constraint).type()).equals(mogram.type()) ||
						((Constraint.Component) constraint).type().equals(mogram.type()) ||
						isOneOf((Constraint.Component) constraint, mogram.type())));
	}

	private List<Constraint.Component> components(List<Constraint> context) {
		return context.stream().filter(c -> c instanceof Constraint.Component).map(c -> (Constraint.Component) c).collect(toList());
	}

	private List<Constraint.MetaFacet> metaFacets(List<Constraint> context) {
		return context.stream().filter(c -> c instanceof Constraint.MetaFacet).map(c -> (Constraint.MetaFacet) c).collect(toList());
	}

	private boolean isOneOf(Constraint.Component allow, String type) {
		if (!(allow instanceof Constraint.OneOf)) return false;
		return ((Constraint.OneOf) allow).components().stream().anyMatch(one -> one.type().endsWith("." + type) || one.type().equals(type));
	}

	private List<Constraint> findInFacets(Mogram mogram) {
		final Mogram context = context(mogram);
		for (Facet facet : context.appliedFacets()) {
			List<Constraint> constraints = language.constraints(facet.fullType());
			if (constraints != null && isComponent(constraints, mogram)) return constraints;
		}
		return null;
	}

	private String simpleType(Constraint.Facet facet) {
		return facet.type().contains(".") ? facet.type().substring(facet.type().lastIndexOf(".") + 1) : facet.type();
	}

	private boolean checkComponentConstraint(Mogram mogram, Constraint constraint) {
		if (!(constraint instanceof Constraint.Component)) return false;
		if (constraint instanceof Constraint.OneOf) return checkAllowOneOf(mogram, constraint);
		return checkAsComponent(mogram, (Constraint.Component) constraint);
	}

	private void checkMetaFacetConstraint(Mogram mogram, Constraint.MetaFacet constraint) {
		if (mogram.type() != null && shortType(mogram.type()).equals(shortType(constraint.type())))
			mogram.type(constraint.type());
	}

	private boolean checkAllowOneOf(Mogram mogram, Constraint allow) {
		for (Constraint.Component one : ((Constraint.OneOf) allow).components()) {
			String absoluteType = one.type();
			if (mogram.type() != null && shortType(mogram.type()).equals(shortType(absoluteType))) {
				mogram.type(absoluteType);
				mogram.metaTypes(language.types(absoluteType));
				resolveFacets(mogram);
				return true;
			}
		}
		return false;
	}

	private boolean checkAsComponent(Mogram mogram, Constraint.Component allow) {
		String absoluteType = allow.type();
		if (mogram.type() != null && shortType(mogram.type()).equals(shortType(absoluteType))) {
			mogram.type(absoluteType);
			mogram.metaTypes(language.types(absoluteType));
			resolveFacets(mogram);
			resolveParameters(mogram);
			return true;
		}
		return false;
	}

	private void resolveFacets(Mogram mogram) {
		for (Facet facet : mogram.appliedFacets()) {
			Constraint.Facet constraint = (Constraint.Facet) language.constraints(mogram.type()).stream().
					filter(c -> c instanceof Constraint.Facet && simpleType((Constraint.Facet) c).equals(facet.type())).findFirst().orElse(null);
			if (constraint != null) {
				facet.fullType(constraint.type());
				List<Constraint.Parameter> parameterConstraints = constraint.constraints().stream().filter(c -> c instanceof Constraint.Parameter).map(c -> (Constraint.Parameter) c).collect(toList());
				resolveParameters(facet.parameters(), parameterConstraints);
			}
		}
	}

	private void resolveParameters(Mogram mogram) {
		List<Constraint.Parameter> constraints = language.constraints(mogram.type()).stream()
				.filter(c -> c instanceof Constraint.Parameter)
				.map(c -> (Constraint.Parameter) c)
				.collect(toList());
		resolveParameters(mogram.parameters(), constraints);
	}

	private static void resolveParameters(List<Parameter> parameters, List<Constraint.Parameter> constraints) {
		for (Parameter param : parameters) {
			Constraint.Parameter constraint = constraints.stream()
					.filter(p -> p.name().equals(param.name()) || p.position() == param.position())
					.findFirst().orElse(null);
			if (constraint != null && param.name().isEmpty()) {
				if (constraint.facet().equals(param.facet()))
					param.name(constraint.name());
			}
		}
	}

	public Mogram context(Mogram mogram) {
		if (mogram == null) return null;
		return mogram.container();
	}
}