package io.intino.tara.dsls;

import io.intino.tara.Language;
import io.intino.tara.language.semantics.*;
import io.intino.tara.language.semantics.constraints.GlobalConstraints;

import java.util.*;
import java.util.stream.Collectors;

import static io.intino.tara.Resolver.shortType;

public abstract class Tara implements Language {
	public static final String GROUP_ID = "io.intino.magritte";
	public static final String ARTIFACT_ID = "framework";
	static final String Root = "";
	private final Map<String, Context> rulesCatalog = new HashMap<>();
	private final Map<String, InstanceContext> declarationsCatalog = new HashMap<>();
	private final List<String> lexicon = new ArrayList<>();

	protected RuleTransaction def(final String qualifiedName) {
		return context -> rulesCatalog.put(qualifiedName, context);
	}

	protected void declare(final String qualifiedName, List<String> types, String path) {
		declarationsCatalog.put(qualifiedName, new InstanceContext(types, path));
	}

	protected Context context(String... types) {
		return new Context(types, new GlobalConstraints().all());
	}

	@Override
	public Map<String, Context> catalog() {
		return rulesCatalog;
	}

	@Override
	public List<Constraint> constraints(String qualifiedName) {
		if (qualifiedName == null || !rulesCatalog.containsKey(qualifiedName)) return null;
		return Collections.unmodifiableList(rulesCatalog.get(qualifiedName).constraints());
	}

	@Override
	public List<Assumption> assumptions(String qualifiedName) {
		if (qualifiedName == null || !rulesCatalog.containsKey(qualifiedName)) return null;
		return Collections.unmodifiableList(rulesCatalog.get(qualifiedName).assumptions());
	}

	@Override
	public Map<String, InstanceContext> instances() {
		return Collections.unmodifiableMap(declarationsCatalog);
	}

	@Override
	public Documentation doc(String qualifiedName) {
		if (qualifiedName == null || !rulesCatalog.containsKey(qualifiedName)) return null;
		return rulesCatalog.get(qualifiedName).doc();
	}

	@Override
	public List<String> types(String qualifiedName) {
		if (qualifiedName == null || !rulesCatalog.containsKey(qualifiedName)) return null;
		return Arrays.asList(rulesCatalog.get(qualifiedName).types());
	}

	@Override
	public String[] lexicon() {
		if (!lexicon.isEmpty()) return lexicon.toArray(new String[0]);
		return calculateLexicon();
	}

	private String[] calculateLexicon() {
		lexicon.addAll(collectTokens());
		return lexicon.toArray(new String[0]);
	}

	private Collection<String> collectTokens() {
		final Set<String> collect = rulesCatalog.keySet().stream().
				filter(qn -> !shortType(qn).isEmpty()).map(t -> {
					final String shortType = shortType(t);
					return shortType.contains(":") ? shortType.substring(0, shortType.indexOf(":")) : shortType;
				}).collect(Collectors.toSet());
		for (Context context : rulesCatalog.values())
			collect.addAll(context.constraints().stream().filter(c -> c instanceof Constraint.Facet).map(c -> ((Constraint.Facet) c).type()).collect(Collectors.toSet()));
		return collect;
	}

	public interface RuleTransaction {
		Context with(Context context);
	}
}
