package io.intino.sumus.reporting.model;

import io.intino.alexandria.Timetag;

import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

public class Period {

	private final Scale scale;
	private final int window;
	private boolean onlyLast;

	public Period(Scale scale) {
		this(scale, 1);
	}

	public Period(Scale scale, int window) {
		this.scale = scale;
		this.window = window;
		this.onlyLast = false;
	}

	public Period onlyLast(boolean onlyLast) {
		this.onlyLast = onlyLast;
		return this;
	}

	public Scale scale() {
		return scale;
	}

	public int window() {
		return window;
	}

	public boolean onlyLast() {
		return onlyLast;
	}

	public String windowString() {
		return String.format("%03d", window);
	}

	public LocalDate startDate(LocalDate date) {
		if (isSingle()) return scale.startDate(date);

		int window = this.window - 1;
		if (scale == Scale.Week) return Scale.WeekCalendar.startDate(date).minus(window, ChronoUnit.WEEKS);
		if (scale == Scale.Month) return date.withDayOfMonth(1).minusMonths(window);
		if (scale == Scale.Quarter) return date.with(date.getMonth().firstMonthOfQuarter()).withDayOfMonth(1).minusMonths((window) * 3L);
		if (scale == Scale.Year) return date.withDayOfYear(1).minusYears(window);
		return date.minusDays(window);
	}

	public List<LocalDate> dates(Timetag timetag) {
		return dates(timetag.datetime().toLocalDate());
	}

	public List<LocalDate> dates(LocalDate date) {
		if (isSingle() && onlyLast()) return isLastDay(date) ? Collections.singletonList(date) : Collections.emptyList();
		if (isSingle()) return scale.dates(date);
		return startDate(date).datesUntil(date.plusDays(1)).collect(Collectors.toList());
	}

	public String format(Timetag timetag) {
		return format(timetag.datetime().toLocalDate());
	}

	public String format(LocalDate date) {
		if (isSingle()) return scale.format(date);
		return scale.format(startDate(date)) + " - " + scale.format(date);
	}

	public String timetag(LocalDate date) {
		return scale.timetag(date);
	}

	public Set<Timetag> timetags(Timetag timetag) {
		return dates(timetag.datetime().toLocalDate()).stream()
				.map(d -> new Timetag(d, timetag.scale()))
				.collect(Collectors.toSet());
	}

	public boolean isSingle() {
		return window == 1;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) return true;
		if (o == null || getClass() != o.getClass()) return false;
		Period period = (Period) o;
		return scale == period.scale && window == period.window;
	}

	@Override
	public int hashCode() {
		return Objects.hash(scale.name(), window);
	}

	public static Period of(String value) {
		if (value.contains(":")) {
			String[] split = value.split(":");
			Scale scale = Scale.valueOf(split[0]);
			String times = split[1].trim();
			return times.equalsIgnoreCase("last") ?
					new Period(scale).onlyLast(true) :
					new Period(scale, Integer.parseInt(times));
		}
		return new Period(Scale.valueOf(value));
	}

	private boolean isLastDay(LocalDate date) {
		LocalDate endDate = scale.endDate(date);
		return date.equals(endDate) || date.equals(LocalDate.now()) || date.equals(LocalDate.now().minusDays(1));
	}
}
