package io.intino.sumus.reporting.loaders;

import io.intino.alexandria.Timetag;
import io.intino.alexandria.logger.Logger;
import io.intino.sumus.engine.Ledger;
import io.intino.sumus.engine.LedgerDecorator;
import io.intino.sumus.engine.ledgers.EmptyLedger;
import io.intino.sumus.engine.ledgers.columnar.ColumnarLedger;
import io.intino.sumus.engine.ledgers.columnar.ColumnarLedgerDecorator;
import io.intino.sumus.engine.ledgers.composite.CompositeLedger;
import io.intino.sumus.model.LedgerDefinition;
import io.intino.sumus.reporting.model.Period;
import io.intino.sumus.reporting.model.Scale;

import java.io.File;
import java.io.IOException;
import java.text.ParseException;
import java.util.HashMap;
import java.util.Map;

import static io.intino.sumus.reporting.loaders.LedgerAggregator.isAggregable;

public class LedgerLoader {

	private final File folder;
	private final LedgerDecorator<ColumnarLedger> decorator;
	private final LedgerAggregator aggregator;

	public LedgerLoader(File folder) {
		this.folder = folder;
		this.decorator = new ColumnarLedgerDecorator(folder);
		this.aggregator = new LedgerAggregator(folder);
	}

	public synchronized Ledger ledger(String name, Scale scale, Timetag timetag) {
		return ledger(name, new Period(scale), timetag);
	}

	public synchronized Ledger ledger(String name, Period period, Timetag timetag) {
		return ledger(name, period.timetags(timetag).toArray(Timetag[]::new));
	}

	public synchronized Ledger ledger(String name, Timetag... timetags) {
		try {
			return loadLedger(name, timetags);
		} catch (LedgerNotFoundException e) {
			return new EmptyLedger();
		} catch (Throwable e) {
			Logger.warn("Error loading ledger: " +  name);
			return new EmptyLedger();
		}
	}

	private Ledger loadLedger(String name, Timetag... timetags) throws IOException, LedgerNotFoundException, ParseException {
		Map<Timetag, Ledger> ledgers = loadLedgers(name, timetags);
		if (ledgers.isEmpty()) throw new LedgerNotFoundException();
		if (ledgers.size() == 1) return ledgers.values().stream().findFirst().orElseThrow(LedgerNotFoundException::new);

		CompositeLedger composite = new CompositeLedger("date");
		ledgers.forEach((timetag, ledger) -> composite.add(ledger, timetag));
		return composite;
	}

	private Map<Timetag, Ledger> loadLedgers(String name, Timetag... timetags) throws IOException, ParseException {
		LedgerDefinition definition = definition(name);
		Map<Timetag, Ledger> ledgers = new HashMap<>();
		for (Timetag timtag : timetags) {
			Ledger ledger = loadLedger(name, definition, timtag);
			if (ledger != null) ledgers.put(timtag, ledger);
		}
		return ledgers;
	}

	private Ledger loadLedger(String name, LedgerDefinition definition, Timetag timetag) throws IOException {
		File file = ledgerFile(name, definition, timetag);
		if (file.exists()) return ledger(definition, file);
		if (isAggregable(definition)) return aggregatedLedger(name, definition, timetag);
		return null;
	}

	private Ledger aggregatedLedger(String name, LedgerDefinition definition, Timetag timetag) {
		try {
			String[][] content = aggregator.aggregate(name, definition, timetag);
			return ledger(definition, content);
		} catch (Throwable e) {
			Logger.error(e);
			return null;
		}
	}

	private Ledger ledger(LedgerDefinition definition, File file) throws IOException {
		ColumnarLedger columnar = new ColumnarLedger(definition).load(file, "\t");
		return isMaster(definition) ? columnar : decorator.decorate(columnar);
	}

	private Ledger ledger(LedgerDefinition definition, String[][] content) {
		return decorator.decorate(new ColumnarLedger(definition).load(content));
	}

	private LedgerDefinition definition(String type) throws ParseException {
		return LedgerDefinition.load(folder, new File(folder, type + ".ledger"));
	}

	private File ledgerFile(String type, LedgerDefinition definition, Timetag timetag) {
		return isMaster(definition) ?
				new File(folder, type + ".tsv") :
				new File(folder, type + "/" + timetag + ".tsv");
	}

	private static boolean isMaster(LedgerDefinition definition) {
		return definition != null && definition.content == LedgerDefinition.Content.Master;
	}

	public static class LedgerNotFoundException extends Exception {
	}
}
