package io.intino.sumus.reporting.loaders;

import io.intino.alexandria.Timetag;
import io.intino.sumus.model.AttributeDefinition;
import io.intino.sumus.model.LedgerDefinition;
import io.intino.sumus.reporting.helpers.ResourceHelper;
import io.intino.sumus.reporting.model.Scale;

import java.io.File;
import java.io.FileNotFoundException;
import java.text.ParseException;
import java.time.LocalDate;
import java.time.Month;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

public class LedgerAggregator {

	private final File root;
	private final String separator;

	public LedgerAggregator(File root) {
		this(root, "\t");
	}

	public LedgerAggregator(File root, String separator) {
		this.root = root;
		this.separator = separator;
	}

	public String[][] aggregate(String ledger, Timetag timetag) throws AggregationException, FileNotFoundException, ParseException {
		return aggregate(ledger, loadDefinition(ledger), timetag);
	}

	public String[][] aggregate(String ledger, LedgerDefinition definition, Timetag timetag) throws AggregationException {
		return isAggregable(definition) ? new Aggregator(ledger, definition, timetag).aggregate() : null;
	}

	private LedgerDefinition loadDefinition(String ledger) throws ParseException, FileNotFoundException {
		File file = new File(root, ledger + ".ledger");
		if (!file.exists()) throw new FileNotFoundException("Ledger definition " + ledger + " not found");
		return LedgerDefinition.load(root, file);
	}

	public static boolean isAggregable(LedgerDefinition definition) {
		LedgerDefinition.Aggregation aggregation = definition.aggregation;
		return aggregation.period != null &&
				aggregation.ledger != null &&
				definition.content != LedgerDefinition.Content.Master;
	}

	private class Aggregator {
		private final String ledger;
		private final Timetag timetag;
		private final LedgerDefinition definition;
		private final LedgerDefinition.Aggregation aggregation;

		public Aggregator(String ledger, LedgerDefinition definition, Timetag timetag) {
			this.ledger = ledger;
			this.definition = definition;
			this.aggregation = definition.aggregation;
			this.timetag = timetag;
		}

		public String[][] aggregate() throws AggregationException {
			return isAggregable(definition) ? aggregateEvents() : null;
		}

		private String[][] aggregateEvents() throws AggregationException {
			return aggregateEvents(keyFunction());
		}

		private String[][] aggregateEvents(Function<String, String> function) {
			return function != null ?
					aggregateFilter(from(), timetag, function) :
					aggregateAll(from(), timetag);
		}

		private String[][] aggregateFilter(Timetag from, Timetag to, Function<String, String> function) {
			Map<String, String> records = new LinkedHashMap<>();
			stream(from, to).forEach(line -> {
				String key = function.apply(line);
				if (key != null) records.put(key, line);
			});
			return toArray(records.values().stream());
		}

		private String[][] aggregateAll(Timetag from, Timetag to) {
			return toArray(stream(from, to));
		}

		private Stream<String> stream(Timetag from, Timetag to) {
			return StreamSupport.stream(from.iterateTo(to).spliterator(), false)
					.map(this::baseLedger)
					.filter(File::exists)
					.flatMap(f -> ResourceHelper.lines(f).stream())
					.filter(l -> l != null && !l.isEmpty());
		}

		private Function<String, String> keyFunction() throws AggregationException {
			String key = aggregation.recordKey;
			if (key == null) return null;
			int index = attributeIndex(key);
			if (index < 0) throw new AggregationException("Aggregation key attribute " + key + " not found in ledger " + ledger);
			return line -> line.split(separator, -1)[index];
		}

		private String[][] toArray(Stream<String> lines) {
			return lines.map(l -> l.split(separator)).toArray(String[][]::new);
		}

		private int attributeIndex(String aggregationKey) {
			List<String> attributes = definition.attributes.stream().map(AttributeDefinition::name).collect(Collectors.toList());
			return attributes.indexOf(aggregationKey);
		}

		private File baseLedger(Timetag timetag) {
			File baseLedgerFolder = new File(root, aggregation.ledger);
			return new File(baseLedgerFolder, timetag + ".tsv");
		}

		private Timetag from() {
			LocalDate date = timetag.datetime().toLocalDate();
			switch (aggregation.period) {
				case Week: return toTimetag(Scale.Week.startDate(date));
				case Month: return toTimetag(Scale.Month.startDate(date));
				case Quarter: return toTimetag(Scale.Quarter.startDate(date));
				case Year: return toTimetag(getYearFrom(date));
				default: return timetag;
			}
		}

		private LocalDate getYearFrom(LocalDate date) {
			Month month = parseMonth(aggregation.from);
			if (month == null) return Scale.Year.startDate(date);

			LocalDate newDate = date.withDayOfMonth(1).withMonth(month.getValue());
			return date.isBefore(newDate) ? newDate.withYear(date.getYear() - 1) : newDate;
		}

		private Timetag toTimetag(LocalDate date) {
			return Timetag.of(date, io.intino.alexandria.Scale.Day);
		}

		private Month parseMonth(String value) {
			return Arrays.stream(Month.values()).filter(m -> m.name().equalsIgnoreCase(value)).findFirst().orElse(null);
		}
	}

	public static class AggregationException extends Exception {
		public AggregationException(String message) {
			super(message);
		}
	}
}