package io.intino.sumus.reporting.helpers;


import io.intino.alexandria.logger.Logger;
import io.intino.sumus.engine.*;
import io.intino.sumus.engine.Cube.Indicator;
import io.intino.sumus.engine.filters.CompositeFilter;
import io.intino.sumus.engine.filters.DateFilter;
import io.intino.sumus.reporting.Dashboard;
import io.intino.sumus.reporting.model.CustomIndicator;

import java.time.LocalDate;
import java.util.*;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

public class CubesHelper {

	static final String FromSubfix = "-from";
	static final String ToSubfix = "-to";

	public static Indicator indicatorOf(Dashboard.Report report, Cube.Cell cell, String name) {
		if (cell == null) return Indicator.None;
		Indicator indicator = formulaIndicator(report, cell, name);
		return indicator != null ? indicator : indicatorOf(cell, name);
	}

	public static Indicator indicatorOf(Cube.Cell cell, String name) {
		if (cell == null) return Indicator.None;
		Indicator indicator = cell.indicator(name);
		return indicator != null ? indicator : Indicator.None;
	}

	private static Indicator formulaIndicator(Dashboard.Report report, Cube.Cell cell, String name) {
		CustomIndicator custom = report.customIndicators().stream().filter(i -> i.name().equalsIgnoreCase(name)).findFirst().orElse(null);
		return custom != null ? formulaIndicator(report, cell, name, custom) : null;
	}

	private static Indicator formulaIndicator(Dashboard.Report report, Cube.Cell cell, String name, CustomIndicator custom) {
		try {
			FormulaCalculator calculator = new FormulaCalculator(report.customIndicators(), cell.indicators());
			Double value = calculator.value(name);
			return new Indicator() {
				@Override
				public String name() {
					return custom.name();
				}

				@Override
				public Object value() {
					return value;
				}

				@Override
				public String units() {
					return custom.units();
				}
			};
		} catch (Throwable e) {
			Logger.error("Error calculating formula " + name, e);
			throw new RuntimeException(e);
		}
	}

	public static Cube emptyCube() {
		return new Cube() {
			@Override
			public List<? extends Dimension> dimensions() {
				return Collections.emptyList();
			}

			@Override
			public List<? extends Cell> cells() {
				return Collections.emptyList();
			}

			@Override
			public Map<String, ? extends Cell> cellMap() {
				return Collections.emptyMap();
			}

			@Override
			public Iterable<Fact> facts(Filter filter) {
				return Collections::emptyIterator;
			}
		};
	}

	public static List<Slice> slices(Ledger ledger, List<String> rawSlices) {
		List<Slice> slices = new ArrayList<>();
		groupByDimension(rawSlices).forEach((dimension, values) -> {
			boolean hasPositives = values.stream().anyMatch(v -> !isNegative(v));
			if (hasPositives) {
				Set<String> positives = values.stream().filter(v -> !isNegative(v)).collect(toSet());
				List<Slice> ledgerSlices = slices(ledger, dimension, positives);
				if (ledgerSlices.isEmpty()) slices.add(null); //None of raw slices have been found on current ledger, still filter
				else slices.addAll(ledgerSlices);
			}
			else {
				Set<String> negatives = values.stream().filter(CubesHelper::isNegative).map(v -> v.substring(1)).collect(toSet());
				slices.addAll(excludedSlices(ledger, dimension, negatives));
			}
		});
		return slices;
	}

	private static Map<String, List<String>> groupByDimension(List<String> slices) {
		return slices.stream()
				.map(s -> s.split(":"))
				.collect(Collectors.groupingBy(d -> d[0], Collectors.mapping(d -> d[1], Collectors.toList())));
	}

	private static List<Slice> slices(Ledger ledger, String dimension, Set<String> values) {
		return values.stream().map(v -> ledger.slice(dimension, v)).filter(Objects::nonNull).collect(toList());
	}

	private static List<Slice> excludedSlices(Ledger ledger, String dimension, Set<String> excluded) {
		if (excluded.isEmpty()) return new ArrayList<>();
		return ledger.dimension(dimension).slices().stream()
				.filter(s -> excluded.stream().noneMatch(e -> e.equalsIgnoreCase(s.name())))
				.collect(toList());
	}

	private static boolean isNegative(String value) {
		return value.startsWith("!");
	}

	public static double doubleValueOf(Indicator indicator) {
		try {
			return Double.parseDouble(indicator.value().toString());
		} catch (Throwable e) {
			return 0;
		}
	}

	public static int intValueOf(Indicator indicator) {
		try {
			return Integer.parseInt(indicator.value().toString());
		} catch (Throwable e) {
			return 0;
		}
	}

	public static Filter dateFilters(Ledger ledger, List<String> slices) {
		Filter[] filters = slices.stream().map(s -> dateFilter(ledger, s)).filter(Objects::nonNull).toArray(Filter[]::new);
		return filters.length > 0 ? CompositeFilter.of(filters) : Filter.None;
	}

	public static DateFilter dateFilter(Ledger ledger, String slice) {
		try {
			if(slice == null || !slice.contains(":")) return null;
			String[] data = slice.split(":");
			String column = data[0];
			LocalDate date = LocalDate.parse(data[1]);
			if (column.endsWith(FromSubfix)) return new DateFilter.FromDateFilter(ledger, column.replace(FromSubfix, ""), date);
			if (column.endsWith(ToSubfix)) return new DateFilter.ToDateFilter(ledger, column.replace(ToSubfix, ""), date);
		}
		catch (Throwable e) {
			Logger.error(e);
		}
		return null;
	}
}
