package io.intino.sumus.reporting.calendars;

import io.intino.alexandria.logger.Logger;

import java.io.File;
import java.nio.file.Files;
import java.time.Duration;
import java.time.LocalDate;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Stream;

public class CustomWeekCalendar extends StandardWeekCalendar {

	private final File file;
	private final Map<LocalDate, String> dayWeek = new TreeMap<>(); //2022-01-07 - 2022|01

	public CustomWeekCalendar(File file) {
		this.file = file;
		reload();
	}

	public void reload() {
		dayWeek.clear();
		try (Stream<String> lines = Files.lines(file.toPath())){
			lines.filter(l -> l != null && !l.isEmpty())
					.map(l -> l.split("\t"))
					.forEach(d -> dayWeek.put(LocalDate.parse(d[1]), d[0]));
		} catch (Throwable e) {
			Logger.error("Custom calendar - error loading map", e);
		}
	}

	public LocalDate startDate(LocalDate date) {
		LocalDate start = customStartDate(date);
		return start != null ? start : super.startDate(date);
	}

	public LocalDate customStartDate(LocalDate date) {
		LocalDate start = null;
		for (LocalDate weekLastDay : dayWeek.keySet()) {
			if (date.isAfter(weekLastDay) && isWithinRange(date, weekLastDay)) start = weekLastDay.plusDays(1);
			else if (date.isBefore(weekLastDay) && isWithinRange(date, weekLastDay)) break;
		}
		return start;
	}

	@Override
	public LocalDate endDate(LocalDate date) {
		LocalDate end = customEndDate(date);
		return end != null ? end : super.endDate(date);
	}

	public LocalDate customEndDate(LocalDate date) {
		for (LocalDate weekLastDay : dayWeek.keySet()) {
			LocalDate nextWeekStart = weekLastDay.plusDays(1);
			if (date.isBefore(nextWeekStart) && isWithinRange(date, weekLastDay)) return weekLastDay;
		}
		return null;
	}

	@Override
	public String format(LocalDate date) {
		String yearWeek = yearWeek(date);
		return yearWeek != null ? year(yearWeek) + "-" + week(yearWeek) : super.format(date);
	}

	@Override
	public String timetag(LocalDate date) {
		String yearWeek = yearWeek(date);
		return yearWeek != null ? year(yearWeek) + "W" + String.format("%02d", week(yearWeek)) : super.timetag(date);
	}

	@Override
	public String label(LocalDate date) {
		String yearWeek = yearWeek(date);
		return yearWeek != null ? "W" + week(yearWeek) : super.label(date);
	}

	private String yearWeek(LocalDate date) {
		LocalDate endDate = customEndDate(date);
		return endDate != null ? dayWeek.get(endDate) : null;
	}

	private int week(String yearWeek) {
		return Integer.parseInt(yearWeek.split("\\|")[1]);
	}

	private int year(String yearWeek) {
		return Integer.parseInt(yearWeek.split("\\|")[0]);
	}

	private boolean isWithinRange(LocalDate d1, LocalDate d2) {
		long days = Duration.between(d1.atStartOfDay(), d2.atStartOfDay()).toDays();
		return Math.abs(days) <= 15;
	}
}
