package io.intino.sumus.reporting.builders;

import io.intino.sumus.engine.Cube;
import io.intino.sumus.engine.Slice;
import io.intino.sumus.reporting.Dashboard;
import io.intino.sumus.reporting.Dashboard.Insight;
import io.intino.sumus.reporting.Node;
import io.intino.sumus.reporting.builders.schemas.PieChart;
import io.intino.sumus.reporting.builders.schemas.PieChart.PieSlice;
import io.intino.sumus.reporting.builders.templates.Renderer;
import io.intino.sumus.reporting.helpers.CubesHelper;

import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;

import static io.intino.sumus.reporting.DashboardBuilder.MaxLevel;
import static io.intino.sumus.reporting.builders.schemas.PieChart.PieSlice.sortMethodOf;

public class PieChartBuilder implements UIBuilder {

	private final Insight insight;
	private final Dashboard.Report report;
	private final MicrositeActionBuilder microsite;

	public PieChartBuilder(Dashboard.Report report, Insight insight, MicrositeActionBuilder microsite) {
		this.report = report;
		this.insight = insight;
		this.microsite = microsite;
	}

	@Override
	public String build(Cube cube, Node node) {
		PieChart pie = new PieChart(insight.id(), insight.label())
				.slices(pieSlices(cube, node));

		return Renderer.render(pie);
	}

	private List<PieSlice> pieSlices(Cube cube, Node node) {
		List<PieSlice> slices = insight.dimensions().length > 0 ? dimensionSlices(cube, node) : indicatorSlices(cube, node);
		return slices.stream()
				.sorted(sortMethodOf(insight.order()))
				.limit(maxItems())
				.collect(Collectors.toList());
	}

	private List<PieSlice> dimensionSlices(Cube cube, Node node) {
		String indicator = insight.indicators()[0];
		List<PieSlice> pieSlices = new LinkedList<>();
		for (Slice slice : slices(cube)) {
			Cube.Cell cell = cube.cell(slice.name());
			if (cell == null) continue;

			String name = cell.toString();
			int value = CubesHelper.intValueOf(CubesHelper.indicatorOf(report, cell, indicator));
			if (name.isEmpty() || value == 0) continue;

			String label = insight.translate(name);
			pieSlices.add(new PieSlice(label, value).onClick(onClickAction(node, cell)));
		}
		return pieSlices;
	}

	private List<PieSlice> indicatorSlices(Cube cube, Node node) {
		List<PieSlice> pieSlices = new LinkedList<>();
		Cube.Cell cell = cube.cell("");
		if (cell == null) return pieSlices;

		for (String indicator : insight.indicators()) {
			int value = CubesHelper.intValueOf(CubesHelper.indicatorOf(report, cell, indicator));
			if (value == 0) continue;

			String label = insight.translate(indicator);
			pieSlices.add(new PieSlice(label, value).onClick(onClickAction(node, cell)));
		}
		return pieSlices;
	}

	private List<Slice> slices(Cube cube) {
		return cube.dimensions().stream()
				.flatMap(d -> d.slices(level()).stream())
				.collect(Collectors.toList());
	}

	private long maxItems() {
		return insight.maxItems() != null ? insight.maxItems() : Integer.MAX_VALUE;
	}

	private int level() {
		return insight.level() != null ? insight.level() : MaxLevel;
	}

	private String onClickAction(Node node, Cube.Cell cell) {
		return microsite.action(node, cell.slices());
	}
}
