package io.intino.sumus.reporting.builders;

import io.intino.alexandria.Timetag;
import io.intino.sumus.engine.Slice;
import io.intino.sumus.reporting.Dashboard.Insight;
import io.intino.sumus.reporting.Dashboard.Report;
import io.intino.sumus.reporting.Node;
import io.intino.sumus.reporting.builders.schemas.MicrositeAction;
import io.intino.sumus.reporting.builders.templates.Renderer;
import io.intino.sumus.reporting.model.Scale;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import static java.util.Arrays.stream;
import static java.util.stream.Collectors.toList;

public class MicrositeActionBuilder {

	private final Insight insight;
	private final Report report;
	private final Timetag timetag;
	private final Scale scale;

	public MicrositeActionBuilder(Insight insight, Report report, Timetag timetag, Scale scale) {
		this.insight = insight;
		this.report = report;
		this.timetag = timetag;
		this.scale = scale;
	}

	public String action(Node node, List<? extends Slice> slices) {
		return action(node, slices.toArray(new Slice[0]));
	}

	public String action(Node node, Slice... slices) {
		List<String> filters = filters(node, slices);
		MicrositeAction action = new MicrositeAction(insight.ledger(), new Timetag(scale.startDate(timetag), timetag.scale()), timetag)
				.filters(filters)
				.dateFilters(dateFilters())
				.title(title())
				.subtitle(subtitle(filters));

		return Renderer.render(action);
	}

	private String title() {
		return report.name() + " " + scale.format(timetag);
	}

	private String subtitle(List<String> filters) {
		return filters.stream()
				.map(f -> f.split(":"))
				.filter(f -> f.length > 1)
				.map(f -> f[1])
				.collect(Collectors.joining(" - "));
	}

	private List<String> dateFilters() {
		return List.of(insight.dateFilters(timetag));
	}

	private List<String> filters(Node node, Slice... slices) {
		List<String> filters = new ArrayList<>();
		if (node.dimension() != null && !isSameDimension(node, slices)) filters.add(node.filter());
		filters.addAll(filters(slices));
		filters.addAll(insightFilters());
		Collections.addAll(filters, report.filters());
		return filters;
	}

	private List<String> filters(Slice... slices) {
		return stream(slices).map(slice -> slice.dimension().name() + ":" + slice.name()).collect(toList());
	}

	private List<String> insightFilters() {
		return stream(insight.filters(timetag)).collect(toList());
	}

	private boolean isSameDimension(Node node, Slice... slices) {
		return stream(slices).anyMatch(s -> isSameDimension(node, s));
	}

	private boolean isSameDimension(Node node, Slice slice) {
		return node.dimension().equalsIgnoreCase(slice.dimension().name());
	}
}
