package io.intino.sumus.reporting.builders;

import io.intino.sumus.engine.Cube;
import io.intino.sumus.engine.Slice;
import io.intino.sumus.reporting.Dashboard;
import io.intino.sumus.reporting.Node;
import io.intino.sumus.reporting.helpers.CubesHelper;
import io.intino.sumus.reporting.helpers.JsonHelper;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static io.intino.sumus.reporting.helpers.FormatHelper.labelByLevel;

public class JsonBuilder implements UIBuilder {

	private final Dashboard.Report report;
	private final Dashboard.View view;

	public JsonBuilder(Dashboard.Report report, Dashboard.View view) {
		this.report = report;
		this.view = view;
	}

	@Override
	public String build(Cube cube, Node node) {
		List<String> records = new ArrayList<>();
		for (Cube.Cell cell : cube.cells()) {
			if (isInvalid(cell) || missingDimensions(cell) || wrongLevel(cell)) continue;

			Map<String, Double> indicatorMap = indicatorValueMap(cell);
			if (areAllValuesZero(indicatorMap)) continue;

			records.add(record(cell, indicatorMap));
		}
		return JsonHelper.array(records);
	}

	private String record(Cube.Cell cell, Map<String, Double> indicatorMap) {
		String[] parameters = Stream.of(label(), slicesAttributes(cell), indicatorsAttributes(indicatorMap))
				.filter(v -> v != null && !v.isEmpty())
				.toArray(String[]::new);
		return JsonHelper.object(parameters);
	}

	private String label() {
		if (view.label() == null) return "";
		return JsonHelper.string("_label") + ":" + JsonHelper.string(view.label());
	}

	private boolean areAllValuesZero(Map<String, Double> indicatorMap) {
		return indicatorMap.isEmpty() || indicatorMap.values().stream().allMatch(i -> i == 0);
	}

	private Map<String, Double> indicatorValueMap(Cube.Cell cell) {
		return Arrays.stream(view.indicators())
				.map(i -> CubesHelper.indicatorOf(report, cell, i))
				.collect(Collectors.toMap(Cube.Indicator::name, i -> round(CubesHelper.doubleValueOf(i))));
	}

	private boolean missingDimensions(Cube.Cell cell) {
		List<String> viewDimensions = Arrays.stream(view.dimensions()).collect(Collectors.toList());
		if (viewDimensions.isEmpty()) return false;

		List<String> cellDimensions = cell.slices().stream().map(s -> s.dimension().name()).collect(Collectors.toList());
		return !cellDimensions.contains(viewDimensions.get(0));
	}

	private boolean wrongLevel(Cube.Cell cell) {
		if (view.level() == null) return false;
		for (Slice slice : cell.slices()) {
			List<? extends Slice> slices = slice.dimension().slices(view.level());
			if (slices.stream().noneMatch(s -> s.name().equalsIgnoreCase(slice.name()))) return true;
		}
		return false;
	}

	private boolean isInvalid(Cube.Cell cell) {
		if (cell == null) return true;
		return cell.slices().isEmpty() && view.dimensions().length > 0;
	}

	private String indicatorsAttributes(Map<String, Double> indicatorMap) {
		return indicatorMap.entrySet().stream()
				.map(e -> JsonHelper.string(e.getKey()) + ":" + e.getValue())
				.collect(Collectors.joining(","));
	}

	private String slicesAttributes(Cube.Cell cell) {
		return Arrays.stream(view.dimensions()).map(dimension -> {
			Slice slice = cell.slices().stream().filter(s -> s.dimension().name().equalsIgnoreCase(dimension)).findFirst().orElse(null);
			String label = slice != null ? labelByLevel(slice.name(), view.level()) : "_All";
			return JsonHelper.string(dimension) + ":" + JsonHelper.string(label);
		})
				.collect(Collectors.joining(","));
	}

	private double round(Double value) {
		try {
			return Math.round(value * 100.0) / 100.0;
		} catch (Throwable e) {
			return 0;
		}
	}
}