package io.intino.sumus.reporting.builders;

import io.intino.sumus.engine.Cube;
import io.intino.sumus.engine.Dimension;
import io.intino.sumus.engine.Slice;
import io.intino.sumus.reporting.Dashboard;
import io.intino.sumus.reporting.Node;
import io.intino.sumus.reporting.builders.schemas.ColumnChart;
import io.intino.sumus.reporting.builders.schemas.ColumnChart.CategoryAction;
import io.intino.sumus.reporting.builders.schemas.ColumnChart.Serie;
import io.intino.sumus.reporting.builders.templates.Renderer;
import io.intino.sumus.reporting.helpers.CubesHelper;
import io.intino.sumus.reporting.helpers.FormatHelper;
import io.intino.sumus.reporting.insights.ColumnInsight;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class ColumnChartBuilder implements UIBuilder {

    private static final int MaxLevel = 100;
    protected final ColumnInsight insight;
    protected final Dashboard.Report report;
    protected final MicrositeActionBuilder microsite;

    public ColumnChartBuilder(Dashboard.Report report, Dashboard.Insight insight, MicrositeActionBuilder microsite) {
        this.report = report;
        this.insight = new ColumnInsight(insight);
        this.microsite = microsite;
    }

    @Override
    public String build(Cube cube, Node node) {
        ColumnChart column = insight.dimensions().length > 0 ?
                indicatorSeries(cube, node) :
                sliceSeries(cube, node);

        return Renderer.render(column);
    }

    private ColumnChart indicatorSeries(Cube cube, Node node) {
        Map<String, Serie> seriesMap = new LinkedHashMap<>();
        List<CategoryAction> actions = new ArrayList<>();

        for (Cube.Cell cell : cells(cube, node)) {
            if (cell == null || cell.toString().isEmpty()) continue;

            String category = insight.translate(cell.toString());

            for (String indicator : insight.indicators()) {
                double value = valueOf(cell, indicator);
                if (isInvalid(value)) continue;

                seriesMap.putIfAbsent(indicator, new Serie(indicator));
                seriesMap.get(indicator).addValue(category, value);
            }
            actions.add(new CategoryAction(category, onClickAction(node, cell)));
        }
        return new ColumnChart(insight.id(), insight.label(), insight.order())
                .showDataLabels(insight.showDataLabels())
                .units(units(cube))
                .stacked(false)
                .plotLines(plotLines())
                .series(new ArrayList<>(seriesMap.values()))
                .onClick(actions);
    }

    private ColumnChart sliceSeries(Cube cube, Node node) {
        List<Serie> series = new ArrayList<>();

        for (Cube.Cell cell : cells(cube, node)) {
            if (cell == null) continue;

            Serie serie = new Serie(insight.translate(cell.toString()))
                    .onClick(onClickAction(node, cell));

            for (String indicator : insight.indicators()) {
                double value = valueOf(cell, indicator);
                if (isInvalid(value)) continue;

                serie.addValue(indicator, value);
            }
            if (isInvalid(serie)) continue;

            series.add(serie);
        }
        return new ColumnChart(insight.id(), insight.label(), insight.order())
                .units(units(cube))
                .plotLines(plotLines())
                .series(series);
    }

    private double valueOf(Cube.Cell cell, String i) {
        Cube.Indicator indicator = CubesHelper.indicatorOf(report, cell, i);
        return FormatHelper.round(CubesHelper.doubleValueOf(indicator), 2);
    }

    private String units(Cube cube) {
        Cube.Cell cell = cube.cell("");
        if (cell == null) return "";

        List<String> units = Arrays.stream(insight.indicators())
                .map(i -> CubesHelper.indicatorOf(report, cell, i))
                .filter(Objects::nonNull)
                .map(Cube.Indicator::units)
                .collect(Collectors.toList());

        if (units.isEmpty() || units.contains(null)) return "";

        String unit = units.get(0);
        return units.stream().allMatch(unit::equalsIgnoreCase) ? unit : "";
    }

    private List<Cube.Cell> cells(Cube cube, Node node) {
        if (cube.dimensions().isEmpty()) return Collections.singletonList(cube.cell(""));
        if (hasCustomSlices()) return customSlicesCells(cube);

        int dimensions = insight.dimensions().length;
        return dimensions == 1 ?
                singleDimensionCells(cube, node) :
                multipleDimensionCells(cube, dimensions);
    }

    private List<Cube.Cell> multipleDimensionCells(Cube cube, int dimensions) {
        List<Cube.Cell> result = new ArrayList<>();
        int level = level();
        for (Cube.Cell cell : cube.cells()) {
            if (!validNotRoot(cell)) continue;
            if (cell.slices().size() != dimensions) continue;
            if (isCustomLevel()) {
                boolean isFromLevel = cell.slices().stream().allMatch(s -> s.level() == level || s.level() < level && !hasChildren(s));
                if (isFromLevel) result.add(cell);
            }
            else result.add(cell);
        }
        return result;
    }

    private List<Cube.Cell> customSlicesCells(Cube cube) {
        return Arrays.stream(insight.slices()).map(cube::cell)
                .filter(ColumnChartBuilder::validNotRoot)
                .collect(Collectors.toList());
    }

    private List<Cube.Cell> singleDimensionCells(Cube cube, Node node) {
        return cube.dimensions().stream()
                .flatMap(d -> sliceStream(d, node))
                .map(cube::cell)
                .filter(ColumnChartBuilder::validNotRoot)
                .collect(Collectors.toList());
    }

    private Stream<String> sliceStream(Dimension dimension, Node node) {
        if (!isNavigable(node, dimension.name())) return dimension.slices(level()).stream().map(Slice::name);
        if (!isCustomLevel()) return node.children().stream().map(Node::name);
        return dimension.slices(level()).stream()
                .map(Slice::name)
                .filter(n -> isChild(node, n))
                .sorted();
    }

    protected List<ColumnChart.PlotLine> plotLines() {
        String definitions = insight.options().get("plot-lines");
        if (definitions == null) return Collections.emptyList();

        return Arrays.stream(definitions.split(","))
                .map(line -> line.split(":"))
                .map(d -> new ColumnChart.PlotLine(
                        Double.parseDouble(d[0]),
                        d.length > 1 ? d[1] : null,
                        d.length > 2 ? d[2] : null)
                )
                .collect(Collectors.toList());
    }

    private String onClickAction(Node node, Cube.Cell cell) {
        return microsite.action(node, cell.slices());
    }

    private boolean isInvalid(Serie serie) {
        return !hasCustomSlices() && serie.isEmpty();
    }

    private boolean hasCustomSlices() {
        return insight.slices().length > 0;
    }

    protected boolean isNavigable(Node node, String dimension) {
        return node.dimension() != null && node.dimension().equalsIgnoreCase(dimension);
    }

    private int level() {
        return insight.level() != null ? insight.level() : MaxLevel;
    }

    private boolean isCustomLevel() {
        return insight.level() != null;
    }

    protected boolean isInvalid(double value) {
        return !insight.showZeros() && !(value > 0);
    }

    private static boolean isChild(Node reference, String node) {
        return reference.isMain() || node.startsWith(reference.name());
    }

    private static boolean hasChildren(Slice slice) {
        return slice.dimension().slices().stream().anyMatch(s -> s.parent() == slice);
    }

    private static boolean validNotRoot(Cube.Cell cell) {
        return cell != null && !cell.toString().isEmpty();
    }
}
