package io.intino.sumus.reporting;

import io.intino.alexandria.Timetag;
import io.intino.alexandria.logger.Logger;
import io.intino.sumus.engine.Cube;
import io.intino.sumus.engine.Dimension;
import io.intino.sumus.engine.Filter;
import io.intino.sumus.engine.Ledger;
import io.intino.sumus.engine.filters.CompositeFilter;
import io.intino.sumus.engine.filters.SliceFilter;
import io.intino.sumus.engine.ledgers.EmptyLedger;
import io.intino.sumus.reporting.Dashboard.Insight;
import io.intino.sumus.reporting.Dashboard.Report;
import io.intino.sumus.reporting.builders.*;
import io.intino.sumus.reporting.helpers.*;
import io.intino.sumus.reporting.loaders.LedgerLoader;
import io.intino.sumus.reporting.loaders.NodeLoader;
import io.intino.sumus.reporting.model.Scale;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.temporal.TemporalAdjusters;
import java.util.*;
import java.util.stream.Collectors;

import static io.intino.sumus.reporting.helpers.ZipHelper.zipAndRemove;
import static java.util.Arrays.stream;
import static java.util.stream.Collectors.toList;

public class DashboardBuilder {
	public static final int MaxLevel = 100;
	private static final String BaseHtml = ResourceHelper.dashboardBaseHtml();

	private final Dashboard dashboard;
	private final File folder;
	private final LedgerLoader ledgerLoader;
	private List<Node> nodes;
	private boolean minify = true;

	public DashboardBuilder(Dashboard dashboard, File folder) {
		this.dashboard = dashboard;
		this.folder = new File(folder, dashboard.name());
		this.folder.mkdirs();
		this.ledgerLoader = new LedgerLoader(dashboard.datamart());
	}

	public Dashboard dashboard() {
		return dashboard;
	}

	public List<Node> nodes() {
		if (nodes == null) nodes = new NodeLoader(dashboard.navigation(), dashboard.nodes()).nodes();
		return nodes;
	}

	public DashboardBuilder minify(boolean minify) {
		this.minify = minify;
		return this;
	}

	public void build(Timetag timetag) {
		build(reportBuilders(timetag));
	}

	public void build(Report report, Timetag timetag) {
		build(reportBuilder(report, timetag));
	}

	private void build(ReportBuilder... builders) {
		stream(builders).forEach(b -> b.build(folder));
	}

	public void buildViews(Report report, Timetag timetag) {
		reportBuilder(report, timetag).buildViews(folder);
	}

	public String buildNodeContent(Report report, Timetag timetag, String nodeName) {
		Node node = nodeOf(nodeName);
		if (node == null) return null;
		ReportBuilder reportBuilder = reportBuilder(report, timetag);
		return reportBuilder.ledgersNotFound() ? null : reportBuilder.bodyOf(node);
	}

	private ReportBuilder[] reportBuilders(Timetag timetag) {
		return dashboard.reports().stream().map(report -> reportBuilder(report, timetag)).toArray(ReportBuilder[]::new);
	}

	private ReportBuilder reportBuilder(Report report, Timetag timetag) {
		return new ReportBuilder(report, timetag);
	}

	private Node nodeOf(String name) {
		return nodes().stream().filter(n -> n.name().equalsIgnoreCase(name)).findFirst().orElse(null);
	}

	private class ReportBuilder {
		public final Report report;
		private final Timetag timetag;
		private final Scale scale;
		private final Map<String, Ledger> ledgers;
		private final List<ViewBuilder> viewBuilders;
		private final List<InsightBuilder> insightBuilders;
		private final NavigationBuilder navBuilder;

		public ReportBuilder(Report report, Timetag timetag) {
			this.report = report;
			this.timetag = timetag;
			this.scale = Scale.of(timetag.scale());
			this.ledgers = loadLedgers();
			this.viewBuilders = loadViewBuilders();
			this.insightBuilders = loadInsightBuilders();
			this.navBuilder = new NavigationBuilder(report);
		}

		public void build(File root) {
			buildViews(root);
			buildReport(root);
		}

		public void buildViews(File root) {
			viewBuilders.forEach(builder -> builder.build(root));
		}

		public void buildReport(File root) {
            if (ledgersNotFound() || isAnyLedgerMissing()) return;

            File reportFolder = folder(root);
			nodes().stream()
					.filter(this::isVisible)
					.forEach(n -> buildReport(reportFolder, n));
			zipAndRemove(reportFolder);
		}

		private void buildReport(File folder, Node node) {
			File file = new File(folder, node.id() + ".html");
			save(file, bodyOf(node));
		}

		public String bodyOf(Node node) {
			String template = template(node);
			for (InsightBuilder b : insightBuilders) {
				if (!template.contains(b.mark())) continue;
				template = template.replace(b.mark(), b.build(node));
			}
			String content = replaceGenerics(template, node);
			return minify ? MinifyHelper.minify(content) : content;
		}

		public File folder(File root) {
			String folderName = report.isSingleton() ? report.name() : scale.timetag(timetag.datetime().toLocalDate());
			return new File(root, "reports/" + report.name() + "/" + folderName);
		}

		private String template(Node node) {
			return dashboard.template() + BaseHtml + navBuilder.build(node) + reportTemplate();
		}

		private String reportTemplate() {
			String template = report.template();
			if (template != null && !template.isEmpty()) return template;
			return insightBuilders.stream().map(InsightBuilder::mark).collect(Collectors.joining("\n"));
		}

		private String replaceGenerics(String template, Node node) {
			LocalDate date = timetag.datetime().toLocalDate();
			TimeNavigatorBuilder timeNavBuilder = new TimeNavigatorBuilder();
			return template
					.replace("::CurrentNode::", node.isMain() ? "Global" : node.name())
					.replace("::CurrentRootNode::", node.isMain() ? "Global" : nodeRootName(node))
					.replace("::CurrentDayTimetag::", Scale.Day.timetag(date))
					.replace("::CurrentDay::", Scale.Day.format(date))
					.replace("::CurrentWeekDay::", Scale.Week.format(date))
					.replace("::CurrentMonth::", Scale.Month.format(date))
					.replace("::CurrentQuarter::", Scale.Quarter.format(date))
					.replace("::CurrentYear::", Scale.Year.format(date))
					.replace("::PreviousYear::", Scale.Year.format(date.minusYears(1)))
					.replace("::NextYear::", Scale.Year.format(date.plusYears(1)))
					.replace("::TimeWidgetDay::", timeNavBuilder.build(timetag, Scale.Day))
					.replace("::TimeWidgetWeek::", timeNavBuilder.build(timetag, Scale.Week))
					.replace("::TimeWidgetMonth::", timeNavBuilder.build(timetag, Scale.Month))
					.replace("::TimeWidgetQuarter::", timeNavBuilder.build(timetag, Scale.Quarter))
					.replace("::TimeWidgetYear::", timeNavBuilder.build(timetag, Scale.Year))
					.replace("::LastMondayDate::", Scale.Day.format(date.with(TemporalAdjusters.previousOrSame(DayOfWeek.MONDAY))));
		}

		private boolean ledgersNotFound() {
			return ledgers.values().stream().allMatch(DashboardBuilder::notFound);
		}

		private boolean isAnyLedgerMissing() {
			return report.requireAllLedgers() && ledgers.values().stream().anyMatch(DashboardBuilder::notFound);
		}

		private boolean isVisible(Node node) {
			return report.visibility().isVisible(node);
		}

		private Map<String, Ledger> loadLedgers() {
			return report.ledgers().stream()
					.collect(HashMap::new, (m, l) -> m.put(l, ledgerLoader.ledger(l, scale, timetag)), HashMap::putAll);
		}

		private List<InsightBuilder> loadInsightBuilders() {
			return report.insights().stream().map(insight -> new InsightBuilder(insight, timetag, scale)).collect(toList());
		}

		private List<ViewBuilder> loadViewBuilders() {
			return report.views().stream().map(insight -> new ViewBuilder(insight, timetag)).collect(toList());
		}


		private class InsightBuilder {
			private final Insight insight;
			private final UIBuilder builder;

			public InsightBuilder(Insight insight, Timetag timetag, Scale scale) {
				this.insight = insight;
				this.builder = UIBuilderFactory.builderOf(report, insight, timetag, scale);
			}

			public String mark() {
				return FormatHelper.translationMark(insight.id());
			}

			public String build(Node node) {
				if (!isVisible(node)) return "";

				Node insightNode = insightNode(node);
				Node dataNode = insightNode != null ? insightNode : node;
				return builder.build(cube(dataNode), dataNode);
			}

			private Cube cube(Node node) {
				Ledger ledger = ledgers.get(insight.ledger());
				return ledger != null && !(ledger instanceof EmptyLedger) ? cube(ledger, node) : CubesHelper.emptyCube();
			}

			private Cube cube(Ledger ledger, Node node) {
				try {
					return ledger.cube()
							.dimensions(dimensions(ledger))
							.filter(filter(ledger, node))
							.build();
				} catch (Throwable e) {
					Logger.error("Error creating cube for " + insight.id() + ". Empty cube returned", e);
					return CubesHelper.emptyCube();
				}
			}

			private List<Dimension> dimensions(Ledger ledger) {
				return stream(insight.dimensions()).map(ledger::dimension).collect(toList());
			}

			private Filter filter(Ledger ledger, Node node) {
				return CompositeFilter.of(
						slicefilterOf(ledger, node, insight.filters(timetag)),
						dateFilterOf(ledger, insight.dateFilters(timetag))
				);
			}

			private boolean isVisible(Node node) {
				return insight.visibility().isVisible(node);
			}

			private Node insightNode(Node current) {
				Node.Type type = insight.node();
				if (current == null || type == null) return null;

				String name = (type == Node.Type.Global) ? Node.GlobalNode :
						(type == Node.Type.Root) ? nodeRootName(current) : null;

				if (name == null) return null;
				return nodes().stream().filter(n -> name.equalsIgnoreCase(n.id())).findFirst().orElse(null);
			}
		}


		private class ViewBuilder {
			private final Dashboard.View view;
			private final Timetag timetag;
			private final Ledger ledger;
			private final UIBuilder builder;

			public ViewBuilder(Dashboard.View view, Timetag timetag) {
				this.view = view;
				this.timetag = timetag;
				this.ledger = ledgerLoader.ledger(view.ledger(), view.period(), timetag);
				this.builder = new JsonBuilder(report, view);
			}

			public void build(File root) {
				if (notFound(ledger)) return;

				File viewFolder = folder(root);
				nodes().forEach(node -> buildView(viewFolder, node));
				zipAndRemove(viewFolder);
			}

			private void buildView(File folder, Node node) {
				String content = bodyOf(node);
				if (JsonHelper.isEmptyArray(content)) return;

				File file = new File(folder, node.id() + ".json");
				save(file, content);
			}

			public String bodyOf(Node node) {
				return builder.build(cube(node), node);
			}

			private Cube cube(Node node) {
				try {
					return ledger.cube()
							.dimensions(dimensions())
							.filter(filter(node))
							.build();
				} catch (Throwable e) {
					return CubesHelper.emptyCube();
				}
			}

			private List<Dimension> dimensions() {
				return stream(view.dimensions()).map(ledger::dimension).collect(toList());
			}

			private Filter filter(Node node) {
				return CompositeFilter.of(
						slicefilterOf(ledger, node, view.filters(timetag)),
						dateFilterOf(ledger, view.dateFilters(timetag))
				);
			}

			public File folder(File root) {
				LocalDate date = timetag.datetime().toLocalDate();
				return new File(root, "views/" + view.name() + "/" + view.period().timetag(date));
			}
		}

		private Filter slicefilterOf(Ledger ledger, Node node, String... sliceNames) {
			List<String> slices = new ArrayList<>();
			Collections.addAll(slices, sliceNames);
			Collections.addAll(slices, report.filters());
			if (node.dimension() != null && !node.name().isEmpty()) slices.add(node.filter());
			return SliceFilter.of(CubesHelper.slices(ledger, slices));
		}

		private Filter dateFilterOf(Ledger ledger, String[] dateFilters) {
			return CubesHelper.dateFilters(ledger, List.of(dateFilters));
		}
	}

	private static String nodeRootName(Node node) {
		return node != null ? node.name().split("\\.")[0] : "";
	}

	protected static boolean notFound(Ledger ledger) {
		return ledger == null || ledger instanceof EmptyLedger;
	}

	private static void save(File file, String content) {
		try {
			file.getParentFile().mkdirs();
			Files.write(file.toPath(), content.getBytes(), StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
		} catch (IOException e) {
			Logger.error(e);
		}
	}
}