package io.intino.sumus.util;

import io.intino.sumus.model.LedgerDefinition;
import io.intino.sumus.parser.SumusGrammar;

import java.text.ParseException;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import static io.intino.sumus.util.Formatters.firstUpperCase;

public class ParseUtils {


	public static SumusGrammar.ValueContext findParameter(List<SumusGrammar.ParameterContext> parameters, String name) throws ParseException {
		if (parameters.isEmpty()) throw new ParseException("Parameter " + name + " not found", 0);
		return findParameterValue(parameters, name).orElseThrow(() -> new ParseException("Parameter " + name + " not found", 0)).value();
	}

	public static SumusGrammar.ValueContext findParameterByNameOrPosition(List<SumusGrammar.ParameterContext> parameters, String name, int pos) throws ParseException {
		if (parameters.isEmpty()) throw new ParseException("Parameter " + name + " not found", 0);
		return findParameterValue(parameters, name).orElse(parameters.get(pos)).value();
	}

	public static Optional<SumusGrammar.ParameterContext> findParameterValue(List<SumusGrammar.ParameterContext> parameters, String name) {
		return parameters.stream()
				.filter(p -> p.IDENTIFIER() != null && name.equals(p.IDENTIFIER().getText()))
				.findFirst();
	}

	public static LedgerDefinition.Content parseContent(SumusGrammar.DeclarationContext content) throws ParseException {
		try {
			return LedgerDefinition.Content.valueOf(firstUpperCase(content.IDENTIFIER().getText()));
		} catch (IllegalArgumentException e) {
			throw new ParseException("Unknown Property 'format' " + content.IDENTIFIER().getText() + ";", 0);
		}
	}

	public static LedgerDefinition.Format parseFormat(SumusGrammar.DeclarationContext content) throws ParseException {
		try {
			return LedgerDefinition.Format.valueOf(firstUpperCase(content.IDENTIFIER().getText()));
		} catch (IllegalArgumentException e) {
			throw new ParseException("Unknown Property 'format' " + content.IDENTIFIER().getText() + ";", 0);
		}
	}

	public static String parseLabel(SumusGrammar.DeclarationContext content) {
		try {
			if (content == null) return null;
			if (content.IDENTIFIER() != null) return content.IDENTIFIER().getText();
			if (content.STRING() != null) return content.STRING().getText().replace("\"", "");
			return null;
		} catch (IllegalArgumentException e) {
			return null;
		}
	}

	public static LedgerDefinition.Aggregation.Period parseAggregationPeriod(String value) {
		try {
			return value != null ? LedgerDefinition.Aggregation.Period.valueOf(firstUpperCase(value)) : null;
		} catch (IllegalArgumentException e) {
			return null;
		}
	}

	public static SumusGrammar.DeclarationContext findDeclaration(List<SumusGrammar.DeclarationContext> declarations, String name) {
		return declarations.stream().filter(s -> s.name().getText().equals(name)).findFirst().orElse(null);
	}

	public static String findDeclarationValue(List<SumusGrammar.DeclarationContext> declarations, String name) {
		SumusGrammar.DeclarationContext declaration = findDeclaration(declarations, name);
		return declaration != null ? declaration.IDENTIFIER().getText() : null;
	}

	public static List<SumusGrammar.DeclarationContext> findSection(SumusGrammar.RootContext rootContext, String name) throws ParseException {
		final SumusGrammar.SectionContext sectionContext = rootContext.section().stream()
				.filter(s -> s.IDENTIFIER().getText().equals(name)).findFirst().orElseThrow(() -> new ParseException(name + " section not found", 0));
		final List<SumusGrammar.DeclarationContext> declarations = sectionContext.body().declaration();
		if (declarations.isEmpty()) throw new ParseException(name + " section not found", 0);
		return declarations;
	}

	public static List<SumusGrammar.DeclarationContext> safeFindSection(SumusGrammar.RootContext rootContext, String name) {
		final SumusGrammar.SectionContext sectionContext = rootContext.section().stream()
				.filter(s -> s.IDENTIFIER().getText().equals(name)).findFirst().orElse(null);
		if(sectionContext == null) return Collections.emptyList();
		final List<SumusGrammar.DeclarationContext> declarations = sectionContext.body().declaration();
		if (declarations.isEmpty()) return Collections.emptyList();
		return declarations;
	}

}
