package io.intino.sumus.model;


import io.intino.sumus.LedgerDefinitionBuilder;
import io.intino.sumus.util.Json;

import java.io.File;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

public class LedgerDefinition {

	public static LedgerDefinition load(File baseDir, File definitionFile) throws ParseException {
		return new LedgerDefinitionBuilder(baseDir).build(definitionFile);
	}

	public static LedgerDefinition load(File baseDir, String definition) throws ParseException {
		return new LedgerDefinitionBuilder(baseDir).build(definition);
	}

	public enum Content {Events, Status, Master}

	public enum Format {Binary, Tsv}

	public final Content content;
	public final Format format;

	public final List<AttributeDefinition> attributes;
	public final List<IndicatorDefinition> indicators;
	public final List<DimensionDefinition> dimensions;

	public Aggregation aggregation;
	public String label;

	public LedgerDefinition(Content content, Format format) {
		this.content = content;
		this.format = format;
		this.attributes = new ArrayList<>();
		this.indicators = new ArrayList<>();
		this.dimensions = new ArrayList<>();
	}

	public LedgerDefinition(Content content, Format format, Aggregation aggregation, String label) {
		this(content, format);
		this.aggregation = aggregation;
		this.label = label;
	}

	public void add(AttributeDefinition attr) {
		attributes.add(attr);
	}

	public void add(IndicatorDefinition indicator) {
		indicators.add(indicator);
	}

	public void add(DimensionDefinition dimension) {
		dimensions.add(dimension);
	}

	public AttributeDefinition attribute(String name) {
		return attributes.stream().filter(a -> a.name().equals(name)).findFirst().orElse(null);
	}

	public IndicatorDefinition indicator(String name) {
		return indicators.stream().filter(a -> a.name().equals(name)).findFirst().orElse(null);
	}

	public List<SimpleIndicatorDefinition> simpleIndicators() {
		return indicators.stream().filter(i -> !i.isComposite()).map(IndicatorDefinition::asSimple).collect(Collectors.toList());
	}

	public List<CompositeIndicatorDefinition> compositeIndicators() {
		return indicators.stream().filter(IndicatorDefinition::isComposite).map(IndicatorDefinition::asComposite).collect(Collectors.toList());
	}

	@Override
	public String toString() {
		return Json.toJsonPretty(this);
	}

	public static class Aggregation {

		public enum Period {Week, Month, Quarter, Year}

		public Period period;
		public String ledger;
		public String recordKey;
		public String from;

		public Aggregation(Period period, String ledger, String recordKey, String from) {
			this.period = period;
			this.ledger = ledger;
			this.recordKey = recordKey;
			this.from = from;
		}
	}
}
