package io.intino.sumus;

import io.intino.sumus.model.AttributeDefinition;
import io.intino.sumus.model.IndicatorDefinition;
import io.intino.sumus.model.LedgerDefinition;
import io.intino.sumus.parser.SumusGrammar;
import io.intino.sumus.parser.SumusGrammar.DeclarationContext;
import io.intino.sumus.parser.SumusParser;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.text.ParseException;
import java.util.List;

import static io.intino.sumus.model.AttributeDefinition.create;
import static io.intino.sumus.util.ParseUtils.*;

public class LedgerDefinitionBuilder {
	static final String CompositeFunction = "calculate";

	private final File baseDirectory;

	public LedgerDefinitionBuilder(File baseDirectory) {
		this.baseDirectory = baseDirectory;
	}

	public LedgerDefinition build(File definition) throws ParseException {
		try {
			return build(Files.readString(definition.toPath(), Charset.defaultCharset()));
		} catch (IOException e) {
			throw new ParseException(e.getMessage(), 0);
		}
	}

	public LedgerDefinition build(InputStream definition) throws ParseException {
		try {
			return build(new String(definition.readAllBytes()));
		} catch (IOException e) {
			throw new ParseException(e.getMessage(), 0);
		}
	}

	public LedgerDefinition build(String definition) throws ParseException {
		final SumusGrammar.RootContext rootContext = new SumusParser(definition).parse();
		LedgerDefinition ledger = createLedger(findSection(rootContext, "properties"));
		fillAttributes(ledger, findSection(rootContext, "attributes"));
		fillIndicators(ledger, safeFindSection(rootContext, "indicators"));
		fillDimensions(ledger, safeFindSection(rootContext, "dimensions"));
		return ledger;
	}

	private void fillAttributes(LedgerDefinition ledger, List<DeclarationContext> attributes) throws ParseException {
		for (DeclarationContext attribute : attributes) {
			AttributeDefinition attr = create(attribute.IDENTIFIER().getText(), attribute.name().getText(), parameters(attribute));
			if (attr == null) throw new ParseException("Error parsing attribute: " + attribute.getText(), 0);
			else ledger.add(attr);
		}
	}

	private String[] parameters(DeclarationContext attribute) {
		if (attribute.parameters() == null) return new String[0];
		return attribute.parameters().parameter().stream()
				.map(p -> p.value().getText().replace("\"", ""))
				.toArray(String[]::new);
	}

	private void fillIndicators(LedgerDefinition ledger, List<DeclarationContext> indicators) throws ParseException {
		IndicatorBuilder simpleBuilder = new IndicatorBuilder();
		CompositeIndicatorBuilder compositeBuilder = new CompositeIndicatorBuilder();
		for (DeclarationContext i : indicators) ledger.add(buildIndicator(ledger, i, simpleBuilder, compositeBuilder));
	}

	private static IndicatorDefinition buildIndicator(LedgerDefinition ledger, DeclarationContext i, IndicatorBuilder simpleBuilder, CompositeIndicatorBuilder compositeBuilder) throws ParseException {
        return CompositeFunction.equals(i.IDENTIFIER().getText()) ?
				compositeBuilder.build(i, ledger) :
				simpleBuilder.build(i, ledger);
	}

	private void fillDimensions(LedgerDefinition ledger, List<DeclarationContext> dimensions) throws ParseException {
		DimensionBuilder builder = new DimensionBuilder(baseDirectory);
		for (DeclarationContext dimension : dimensions) ledger.add(builder.build(dimension, ledger));
	}

	private LedgerDefinition createLedger(List<DeclarationContext> declarations) throws ParseException {
		final DeclarationContext content = findDeclaration(declarations, "content");
		if (content == null) throw new ParseException("Property 'content' not found", 0);
		final DeclarationContext format = findDeclaration(declarations, "format");
		if (format == null) throw new ParseException("Property 'format' not found", 0);
		final DeclarationContext label = findDeclaration(declarations, "label");

		return new LedgerDefinition(parseContent(content), parseFormat(format), createAggregationDefinition(declarations), parseLabel(label));
	}

	private static LedgerDefinition.Aggregation createAggregationDefinition(List<DeclarationContext> declarations) {
		return new LedgerDefinition.Aggregation(
				parseAggregationPeriod(findDeclarationValue(declarations, "aggregation")),
				findDeclarationValue(declarations, "aggregation-ledger"),
				findDeclarationValue(declarations, "aggregation-key"),
				findDeclarationValue(declarations, "aggregation-from"));
	}
}