package io.intino.sumus;

import io.intino.sumus.model.AttributeDefinition;
import io.intino.sumus.model.IndicatorDefinition.Scale;
import io.intino.sumus.model.LedgerDefinition;
import io.intino.sumus.model.SimpleIndicatorDefinition;
import io.intino.sumus.model.SimpleIndicatorDefinition.Function;
import io.intino.sumus.parser.SumusGrammar;
import io.intino.sumus.parser.SumusGrammar.DeclarationContext;
import io.intino.sumus.parser.SumusGrammar.ParameterContext;
import io.intino.sumus.parser.SumusGrammar.ValueContext;
import io.intino.sumus.util.ParseUtils;

import java.text.ParseException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import static io.intino.sumus.util.ParseUtils.findParameterByNameOrPosition;
import static java.lang.Double.parseDouble;

public class IndicatorBuilder {

	public SimpleIndicatorDefinition build(DeclarationContext i, LedgerDefinition ledger) throws ParseException {
		SumusGrammar.ValueContext attr = findParameterByNameOrPosition(i.parameters().parameter(), "attribute", 0);
		AttributeDefinition attribute = ledger.attribute(attr.getText());
		if (attribute == null)
			throw new ParseException("Attribute '" + attr.getText() + "' not found for: " + i.name().getText(), 0);
		return createIndicator(i, attribute);
	}

	private SimpleIndicatorDefinition createIndicator(DeclarationContext i, AttributeDefinition attr) throws ParseException {
		String name = i.name().getText();
		Function function = parseFunction(i.IDENTIFIER().getText());
		String unit = parseUnit(findParameter(i.parameters().parameter(), "unit"));
		Scale scale = parseScale(findParameter(i.parameters().parameter(), "scale"));
		List<String> values = parseValues(findParameter(i.parameters().parameter(), "values"));
		int decimals = parseDecimals(findParameter(i.parameters().parameter(), "decimals"));
		return new SimpleIndicatorDefinition(name, function, attr, unit, scale, values, decimals);
	}

	private Function parseFunction(String function) throws ParseException {
		return Arrays.stream(Function.values())
				.filter(f -> f.label.equals(function)).findFirst()
				.orElseThrow(() -> new ParseException("Formula not found '" + function + "'", 0));
	}

	public static Scale parseScale(ValueContext scale) throws ParseException {
		if (scale == null) return null;
		if (scale.range().size() != 1)
			throw new ParseException("Scale '" + scale.getText() + "' not well formatted", 0);
		List<SumusGrammar.IntegerValueContext> values = scale.range().get(0).integerValue();
		return new Scale(parseDouble(values.get(0).getText()), parseDouble(values.get(1).getText()));
	}

	private List<String> parseValues(ValueContext value) {
		if (value == null) return Collections.emptyList();
		return value.STRING().stream().map(s -> cleanString(s.getText())).collect(Collectors.toUnmodifiableList());
	}


	public static String parseUnit(ValueContext unit) throws ParseException {
		if (unit == null) return null;
		if (unit.STRING().size() != 1)
			throw new ParseException("Unit '" + unit.getText() + "' not well formatted", 0);
		return unit.STRING().get(0).getText().replace("\"", "");
	}

	public static Integer parseDecimals(ValueContext decimals) throws ParseException {
		if (decimals == null) return 0;
		if (decimals.integerValue().size() != 1)
			throw new ParseException("Decimals '" + decimals.integerValue() + "' not well formatted", 0);
		try {
			return Integer.parseInt(decimals.integerValue().get(0).getText());
		}
		catch (Throwable e) {
			return 0;
		}
	}

	private static ValueContext findParameter(List<ParameterContext> parameters, String name) {
		try {
			return ParseUtils.findParameter(parameters, name);
		} catch (ParseException ignored) {
			return null;
		}
	}

	private String cleanString(String value) {
		return value.replace("\"", "");
	}
}
