package io.intino.sumus;

import io.intino.sumus.model.AttributeDefinition;
import io.intino.sumus.model.DimensionDefinition;
import io.intino.sumus.model.DimensionDefinition.Categorical;
import io.intino.sumus.model.DimensionDefinition.Numerical;
import io.intino.sumus.model.LedgerDefinition;
import io.intino.sumus.parser.*;
import io.intino.sumus.parser.SumusGrammar.DeclarationContext;
import io.intino.sumus.parser.SumusGrammar.IntegerValueContext;
import org.antlr.v4.runtime.RuleContext;
import org.antlr.v4.runtime.tree.TerminalNode;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.text.ParseException;
import java.util.*;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import static io.intino.sumus.util.ParseUtils.findParameter;
import static io.intino.sumus.util.ParseUtils.findParameterByNameOrPosition;
import static java.util.Objects.isNull;
import static java.util.regex.Pattern.compile;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

public class DimensionBuilder {
	private final File baseDirectory;

	public DimensionBuilder(File baseDirectory) {
		this.baseDirectory = baseDirectory;
	}

	public DimensionDefinition build(DeclarationContext d, LedgerDefinition ledger) throws ParseException {
		SumusGrammar.ValueContext attr = findParameterByNameOrPosition(d.parameters().parameter(), "attribute", 0);
		AttributeDefinition attribute = ledger.attribute(attr.getText());
		if (attribute == null)
			throw new ParseException("Attribute '" + attr.getText() + "' not found for: " + d.name(), 0);
		String type = d.IDENTIFIER().getText();
		if ("numerical".equals(type)) return createNumerical(d, attribute);
		else if ("categorical".equals(type)) return createCategorical(d, attribute);
		else throw new ParseException("Unknown dimension type: " + type, 0);
	}

	private DimensionDefinition createNumerical(DeclarationContext dimension, AttributeDefinition attribute) throws ParseException {
		return new Numerical(dimension.name().getText(), attribute, createNumericalCategories(dimension, attribute));
	}

	private DimensionDefinition createCategorical(DeclarationContext dimension, AttributeDefinition attribute) throws ParseException {
		return new Categorical(dimension.name().getText(), attribute, createCategoricalCategories(dimension));
	}

	private Map<String, Predicate<?>> createNumericalCategories(DeclarationContext dimension, AttributeDefinition attribute) throws ParseException {
		try {
			return mapRanges(findParameter(dimension.parameters().parameter(), "ranges").integerValue(), attribute);
		} catch (ParseException ignored) {
		}
		return mapNumericalSlices(clean(findParameter(dimension.parameters().parameter(), "slices").STRING()), attribute);
	}

	private Map<String, Predicate<?>> createCategoricalCategories(DeclarationContext dimension) throws ParseException {
		try {
			return mapCategoricalSlices(clean(findParameter(dimension.parameters().parameter(), "slices").STRING()));
		} catch (ParseException ignored) {
		}
		return mapFromResource(clean(findParameter(dimension.parameters().parameter(), "from").STRING()).get(0));
	}

	private Map<String, Predicate<?>> mapNumericalSlices(List<String> slices, AttributeDefinition attribute) throws ParseException {
		Map<String, Predicate<?>> categories = new LinkedHashMap<>();
		for (String slice : slices) {
			String[] keyValue = parseSlice(slice);
			Matcher matcher = rangePattern(keyValue[1]);
			if (!matcher.matches())
				throw new ParseException("Slice not well formatted. Pattern didn't match. " + slice, 0);
			categories.put(keyValue[0], numericalSlicePredicate(matcher, keyValue[1], attribute));
		}
		return categories;
	}

	private Predicate<?> numericalSlicePredicate(Matcher matcher, String slice, AttributeDefinition attribute) throws ParseException {
		if (slice.startsWith("[") || matcher.group(2).equals("~"))
			return leftClosedPredicate(attribute, parse(matcher.group(2)), parse(matcher.group(5)));
		if (slice.endsWith("]") || matcher.group(5).equals("~"))
			return rightClosedPredicate(attribute, parse(matcher.group(0)), parse(matcher.group(1)));
		else
			throw new ParseException("Slice not well formatted. Missing closed interval. " + slice, 0);
	}

	private Map<String, Predicate<?>> mapCategoricalSlices(List<String> slices) throws ParseException {
		Map<String, Predicate<?>> categories = new LinkedHashMap<>();
		for (String slice : slices) {
			String[] keyValue = parseSlice(slice);
			categories.put(keyValue[0], compile(keyValue[1]).asMatchPredicate());
		}
		return categories;
	}

	private Map<String, Predicate<?>> mapFromResource(String resource) throws ParseException {
		Map<String, Predicate<?>> categories = new LinkedHashMap<>();
		File file = new File(baseDirectory, resource);
		if (!file.exists()) throw new ParseException("Resource " + file.getAbsolutePath() + " not found", 0);
		try {
			List<String> lines = Files.readAllLines(file.toPath());
			lines.stream().map(l -> l.split("\t")).forEach(l -> categories.put(l[0], compile(l[1]).asMatchPredicate()));
			return categories;
		} catch (IOException e) {
			throw new ParseException("Resource " + file.getAbsolutePath() + "cannot be loaded", 0);
		}
	}

	private String[] parseSlice(String slice) throws ParseException {
		String[] keyValue = slice.split("\\|");
		if (keyValue.length != 2)
			throw new ParseException("Slice not well formatted. Should have '|' separator. " + slice, 0);
		return keyValue;
	}

	private List<String> clean(List<TerminalNode> node) {
		return node.stream().map(n -> cleanString(n.getText())).collect(Collectors.toList());
	}

	private String cleanString(String value) {
		return value.replace("\"", "");
	}

	private Map<String, Predicate<?>> mapRanges(List<IntegerValueContext> ranges, AttributeDefinition attr) {
		Long[] longs = asLongs(ranges.stream().map(RuleContext::getText));
		List<String> categories = categoriesOf(longs);
		return IntStream
				.range(0, categories.size())
				.boxed()
				.collect(toMap(categories::get, pos -> leftClosedPredicate(attr, longs[pos], longs[pos + 1]), (a, b) -> b, LinkedHashMap::new));
	}

	private Predicate<?> leftClosedPredicate(AttributeDefinition attr, Long left, Long right) {
		return attr.isReal() ?
				doublePredicate(v -> isNull(left) || toDouble(v) >= left, v -> isNull(right) || toDouble(v) < right) :
				longPredicate(v -> isNull(left) || toLong(v) >= left, v -> isNull(right) || toLong(v) < right);
	}

	private Predicate<?> rightClosedPredicate(AttributeDefinition attr, Long left, Long right) {
		return attr.isReal() ?
				doublePredicate(v -> isNull(left) || toDouble(v) > left, v -> isNull(right) || toDouble(v) <= right) :
				longPredicate(v -> isNull(left) || toLong(v) > left, v -> isNull(right) || toLong(v) <= right);
	}

	private Predicate<Object> longPredicate(Predicate<Object> left, Predicate<Object> right) {
		return v -> !isNull(v) && left.test(toLong(v)) && right.test(toLong(v));
	}

	private Predicate<Object> doublePredicate(Predicate<Object> left, Predicate<Object> right) {
		return v -> !isNull(v) && left.test(toDouble(v)) && right.test(toDouble(v));
	}

	private long toLong(Object v) {
		return ((Number)v).longValue();
	}

	private double toDouble(Object v) {
		return ((Number)v).doubleValue();
	}

	private Long[] asLongs(Stream<String> options) {
		return options.map(DimensionBuilder::parse)
				.toArray(Long[]::new);
	}

	private Matcher rangePattern(String range) {
		String patternStr = "([\\[(])((\\d+)|~),\\s?((\\d+)|~)(]|\\))";
		Pattern pattern = compile(patternStr);
		return pattern.matcher(range);
	}

	private List<String> categoriesOf(Long[] longs) {
		return IntStream.range(0, longs.length - 1)
				.mapToObj(i -> clean("[" + longs[i] + ".." + longs[i + 1] + "]"))
				.collect(toList());
	}

	private static Long parse(String s) {
		try {
			if(s == null) return null;
			return Long.parseLong(s.trim());
		} catch (NumberFormatException e) {
			return null;
		}
	}

	private static String clean(String value) {
		return value.replace("null", "");
	}
}