package io.intino.sumus.engine;

import io.intino.sumus.engine.filters.SliceFilter;

import java.util.List;
import java.util.Map;
import java.util.function.Predicate;

import static java.util.stream.Collectors.toList;

public interface Cube {
	List<? extends Dimension> dimensions();

	default List<Slice> slices() {
		return cells().stream().flatMap(c->c.slices().stream()).distinct().collect(toList());
	}

	List<? extends Cell> cells();
	Map<String, ? extends Cell> cellMap();

	default List<Cell> cells(Predicate<Cell> predicate) {
		return cells().stream().filter(predicate).collect(toList());
	}

	default Cell cell(String name) {
		return cellMap().get(name);
	}

	Iterable<Fact> facts(Filter filter);

	default Iterable<Fact> facts(Slice... slices) {
		return facts(SliceFilter.of(slices));
	}

	default Iterable<Fact> facts() {
		return facts(Filter.None);
	}

	interface Cell {
		List<? extends Slice> slices();
		List<? extends Indicator> indicators();
		Map<String, ? extends Indicator> indicatorsMap();
		default Indicator indicator(String name) {
			return indicatorsMap().get(name);
		}
		Iterable<Fact> facts();

	}

	interface Indicator {
		Indicator None = indicator("",0);

		String name();
		Object value();
		String units();

		default Indicator as(String name, String units, double ratio) {
			return ratio == 1 ? indicator(name, value(), units) : indicator(name, ratio(value(), ratio), units);
		}
	}

	static Object ratio(Object value, double ratio) {
		if (value instanceof Integer) return (Integer) value * ratio;
		if (value instanceof Long) return (Long) value * ratio;
		if (value instanceof Double) return (Double) value * ratio;
		return value;
	}

	static Indicator indicator(String name, Object value) {
		return indicator(name, value, "");
	}

	static Indicator indicator(String name, Object value, String units) {
		return new Indicator() {
			@Override
			public String name() {
				return name;
			}

			@Override
			public Object value() {
				return value;
			}

			@Override
			public String units() {
				return units;
			}

			@Override
			public String toString() {
				return name;
			}
		};
	}

}
