package io.intino.sumus.engine.ledgers.columnar;

import io.intino.sumus.LedgerDefinitionBuilder;
import io.intino.sumus.engine.Attribute;
import io.intino.sumus.engine.Fact;
import io.intino.sumus.engine.Ledger;
import io.intino.sumus.engine.LedgerDecorator;
import io.intino.sumus.model.AttributeDefinition;
import io.intino.sumus.model.LedgerDefinition;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

public class ColumnarLedgerDecorator implements LedgerDecorator<ColumnarLedger> {

    private static final String ID = "id";
    private static final String TAB = "\t";
    private static final String TSV_EXTENSION = ".tsv";
    public static final String LEDGER_EXTENSION = ".ledger";

    private final File root;
    private final Map<String, Master> masters = new HashMap<>();

    public ColumnarLedgerDecorator(File root) {
        this.root = root;
    }

    @Override
    public ColumnarLedger decorate(ColumnarLedger ledger) {
        ColumnarLedger result = new ColumnarLedger(ledger.definition);
        mastersOf(ledger.definition()).forEach(master -> join(ledger, master, result));
        ledger.columns.forEach(result::add);
        result.removeColumnIf(c -> c.type() == AttributeDefinition.Type.key);
        return result;
    }

    private void join(ColumnarLedger ledger, Master master, ColumnarLedger result) {
        for(Column column : master.columns(ledger)) result.add(column);
    }

    private Stream<Master> mastersOf(LedgerDefinition definition)  {
        return masterLedgersDeclarationsOf(definition)
                .peek(this::loadMasterIfNotExists)
                .map(masters::get);
    }

    private void loadMasterIfNotExists(String name) {
        try {
            if (masters.containsKey(name)) return;
            ColumnarLedger ledger = decorate(ledgerOf(ledgerDefinitionOf(name)));
            read(ledger, new File(root, name + TSV_EXTENSION), TAB);
            masters.put(name, new Master(ledger));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private ColumnarLedger ledgerOf(LedgerDefinition definition) {
        return new ColumnarLedger(definition);
    }

    private void read(Ledger ledger, File file, String separator) throws IOException {
        if (ledger instanceof ColumnarLedger) ((ColumnarLedger) ledger).load(file, separator);
    }

    private LedgerDefinition ledgerDefinitionOf(String name) {
        try {
            return new LedgerDefinitionBuilder(root).build(new File(root, name + LEDGER_EXTENSION));
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private Stream<String> masterLedgersDeclarationsOf(LedgerDefinition definition) {
        return definition.attributes.stream()
                .filter(a -> a.type() == AttributeDefinition.Type.key)
                .map(a -> (AttributeDefinition.Key) a)
                .map(AttributeDefinition.Key::ledgerJoin)
                .filter(a -> !a.isEmpty());
    }

    private static class Master {

        private final ColumnarLedger ledger;
        private final Map<String, Fact> facts;

        public Master(ColumnarLedger ledger) {
            this.ledger = ledger;
            this.facts = new HashMap<>(ledger.size());
            ledger.facts().forEach(f -> facts.put(idOf(f), f));
        }

        public Fact get(Fact fact) {
            return get(idOf(fact));
        }

        public Fact get(String id) {
            return facts.get(id);
        }

        public Attribute[] attributes() {
            return ledger.attributes().stream().filter(a -> a.type() != AttributeDefinition.Type.key).toArray(Attribute[]::new);
        }

        private String idOf(Fact f) {
            return String.valueOf(f.value(ID));
        }

        public MasterColumn[] columns(ColumnarLedger ledger) {
            Column foreign = ledger.column(ID);
            return this.ledger.columns.stream()
                    .map(c -> new MasterColumn(this, c, foreign))
                    .toArray(MasterColumn[]::new);
        }
    }

    public static class MasterColumn implements Column {

        private final Master master;
        private final Column primary;
        private final Column foreign;

        public MasterColumn(Master master, Column primary, Column foreign) {
            this.master = master;
            this.primary = primary;
            this.foreign = foreign;
        }

        @Override
        public Attribute attribute() {
            return primary.attribute();
        }

        @Override
        public String name() {
            return primary.name();
        }

        @Override
        public AttributeDefinition.Type type() {
            return primary.type();
        }

        @Override
        public boolean hasNA() {
            return foreign.hasNA() || primary.hasNA();
        }

        @Override
        public Stream<?> uniques() {
            return primary.uniques();
        }

        @Override
        public Object min() {
            return primary.min();
        }

        @Override
        public Object max() {
            return primary.max();
        }

        @Override
        public int size() {
            return foreign.size();
        }

        @Override
        public Object value(int idx) {
            Fact fact = master.get(idOf(idx));
            return fact != null ? fact.value(attribute()) : null;
        }

        private String idOf(int idx) {
            return (String) foreign.value(idx);
        }

        @Override
        public String toString() {
            return type().name() + " " + name();
        }
    }
}
