package io.intino.sumus.engine.io;

import org.xerial.snappy.SnappyInputStream;

import java.io.*;
import java.nio.file.Files;
import java.util.stream.Stream;
import java.util.zip.ZipInputStream;

public class TsvLedgerReader {

    public static String[][] read(File file, String separator) throws IOException {
        if(file.getName().endsWith(".tsv")) return readUncompressedTsv(file, separator);
        if(file.getName().endsWith(".sz")) return readSnappyTsv(file, separator);
        if(file.getName().endsWith(".zip")) return readZipTsv(file, separator);
        return readUncompressedTsv(file, separator);
    }

    private static String[][] readZipTsv(File file, String separator) throws IOException {
        try(ZipInputStream zis = new ZipInputStream(new FileInputStream(file))) {
            return parseTsvLines(separator, linesFrom(readZipTsv(zis)));
        }
    }

    private static Stream<String> linesFrom(byte[] uncompressed) {
        return new BufferedReader(new InputStreamReader(new ByteArrayInputStream(uncompressed))).lines();
    }

    private static byte[] readZipTsv(ZipInputStream zis) throws IOException {
        zis.getNextEntry();
        byte[] buffer = new byte[1024];
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream(buffer.length);
        int read;
        while ((read = zis.read(buffer)) > 0) {
            outputStream.write(buffer, 0, read);
        }
        return outputStream.toByteArray();
    }

    private static String[][] readSnappyTsv(File file, String separator) throws IOException {
        try(BufferedReader reader = new BufferedReader(new InputStreamReader(new SnappyInputStream(new FileInputStream(file))))) {
            return parseTsvLines(separator, reader.lines());
        } catch (IOException e) {
            throw new IOException("File is not a valid (snappy) compressed file", e);
        }
    }

    private static String[][] readUncompressedTsv(File file, String separator) throws IOException {
        try(Stream<String> lines = Files.lines(file.toPath())) {
            return parseTsvLines(separator, lines);
        }
    }

    private static String[][] parseTsvLines(String separator, Stream<String> lines) {
        return lines.map(l -> l.split(separator)).toArray(String[][]::new);
    }
}
