package io.intino.sumus.engine.builders.accumulators;

import io.intino.sumus.engine.Cube;

import java.util.Arrays;
import java.util.List;

import static io.intino.sumus.engine.Cube.indicator;

public class DoubleAccumulator extends BuilderAccumulator {

	private final int[] count = new int[threadCount];
	private final double[] sum = new double[threadCount];
	private final double[] totalSum = new double[threadCount];
	private final double[] min = new double[threadCount];
	private final double[] max = new double[threadCount];

	public DoubleAccumulator(String name) {
		super(name);
		Arrays.fill(this.min, Double.MAX_VALUE);
		Arrays.fill(this.max, Double.MIN_VALUE);
	}

	@Override
	public void addTotal(Object value) {
		final int t = threadIndex();
		total[t]++;
		if (value != null) totalSum[t] += (double) value;
	}

	public void add(Object value) {
		if (value == null) return;
		final int t = threadIndex();
		final double v = (double) value;
		this.count[t]++;
		this.sum[t] += v;
		this.min[t] = Math.min(this.min[t], v);
		this.max[t] = Math.max(this.max[t], v);
	}

	@Override
	public List<Cube.Indicator> indicators() {
		int count = count();
		double totalSum = totalSum();
		double sum = sum();
		double min = min();
		double max = max();

		return List.of(
				indicator("count(" + name + ")", count),
				indicator("ratio(" + name + ")", totalSum > 0 ? sum / totalSum : 0),
				indicator("sum(" + name + ")", sum),
				indicator("average(" + name + ")", count > 0 ? sum / count : 0),
				indicator("min(" + name + ")", min),
				indicator("max(" + name + ")", max)
		);
	}

	public double max() {
		return Arrays.stream(this.max).max().orElse(Double.MIN_VALUE);
	}

	public double min() {
		return Arrays.stream(this.min).min().orElse(Double.MAX_VALUE);
	}

	public double sum() {
		return Arrays.stream(this.sum).sum();
	}

	public double totalSum() {
		return Arrays.stream(this.totalSum).sum();
	}

	public int count() {
		return Arrays.stream(this.count).sum();
	}
}
