package io.intino.sumus.engine.ledgers.columnar.columns;

import io.intino.sumus.engine.Attribute;
import io.intino.sumus.engine.dimensions.Category;
import io.intino.sumus.engine.ledgers.columnar.Column;
import io.intino.sumus.model.AttributeDefinition;

import java.time.LocalDate;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.function.Function;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;

public class DataColumn implements Column {

    public final Attribute attribute;
    public final List<Object> values;
    private final Map<String, Object> cache;
    private Object min;
    private Object max;

    public DataColumn(Attribute attribute, String... values) {
        this.attribute = attribute;
        this.cache = new HashMap<>(values.length / 2);
        this.values = map(values, function(values));
    }

    private Comparator<Object> comparatorOf(AttributeDefinition.Type type) {
        switch (type) {
            case int8:
            case int16:
            case int32:
            case int64:
            case date:
            case time:
                return Comparator.comparingLong(a -> (long) a);
            case real32:
            case real64:
                return Comparator.comparingDouble(a -> (double) a);
        }
        return (a, b) -> 0;
    }

    private Function<String, Object> function(String[] values) {
        switch (attribute.type()) {
            case category:
                return get(categoriesIn(values));
            case int8:
            case int16:
            case int32:
            case int64:
                return s -> computeIfAbsent(s, Long::parseLong);
            case real32:
            case real64:
                return s -> computeIfAbsent(s, Double::parseDouble);
            case date:
                return s -> computeIfAbsent(s, DataColumn.this::toEpoch);
            case time:
                return s -> computeIfAbsent(s, DataColumn.this::toSecondDay);
        }
        return s -> s;
    }

    private Object computeIfAbsent(String value, Function<String, Object> function) {
        if (cache.containsKey(value)) return cache.get(value);
        Object result = function.apply(value);
        cache.put(value, result);
        return result;
    }

    private Function<String, Object> get(Map<String, Category> categories) {
        return categories::get;
    }

    private List<Object> map(String[] values, Function<String, Object> function) {
        return Arrays.stream(values)
                .map(DataColumn::clean)
                .map(v -> v.isEmpty() ? null : catching(() -> function.apply(v)))
                .collect(toList());
    }

    private static String clean(String v) {
        return v != null ? v.trim() : "";
    }

    private static Map<String, Category> categoriesIn(String[] values) {
        Map<String, Category> categories = new HashMap<>();
        for (String value : values) update(categories, split(clean(value)));
        return categories;
    }

    private static List<String> split(String value) {
        if (value.isEmpty()) return emptyList();
        List<String> result = new ArrayList<>();
        int i = value.indexOf('.');
        while (i > 0) {
            result.add(value.substring(0, i));
            i = value.indexOf('.', i + 1);
        }
        result.add(value);
        return result;
    }

    private static void update(Map<String, Category> categories, List<String> labels) {
        Category parent = null;
        for (String label : labels) {
            if (!categories.containsKey(label))
                categories.put(label, new Category(categories.size() + 1, label, parent));
            parent = categories.get(label);

        }
    }

    private void setRange() {
        Comparator<Object> comparator = comparatorOf(attribute.type());
        for (Object value : values) {
            if (value == null) continue;
            if (min == null || comparator.compare(value, min) < 0) min = value;
            if (max == null || comparator.compare(value, max) > 0) max = value;
        }
    }

    @Override
    public String name() {
        return attribute.name();
    }

    @Override
    public AttributeDefinition.Type type() {
        return attribute.type();
    }

    public int size() {
        return values.size();
    }

    @Override
    public Attribute attribute() {
        return attribute;
    }

    @Override
    public Object value(int idx) {
        return values.get(idx);
    }

    @Override
    public boolean hasNA() {
        for (Object value : values)
            if (value == null) return true;
        return false;
    }

    @Override
    public List<Object> uniques() {
        return values.stream()
                .distinct()
                .filter(Objects::nonNull)
                .collect(toList());
    }

    @Override
    public Object min() {
        if (min == null) setRange();
        return min;
    }

    @Override
    public Object max() {
        if (max == null) setRange();
        return max;
    }

    private Object catching(Callable<Object> function) {
        try {
            return function.call();
        } catch (Exception e) {
            return null;
        }
    }

    private static final DateTimeFormatter DateFormatter = DateTimeFormatter.ofPattern("yyyyMMdd");
    private static final DateTimeFormatter TimeFormatter = DateTimeFormatter.ofPattern("HHmmss");

    private Object toEpoch(String value) {
        if (value == null) return null;
        return LocalDate.parse(trim(value), DateFormatter).toEpochDay();
    }

    private Object toSecondDay(String value) {
        if (value == null) return null;
        return LocalTime.parse(trim(value), TimeFormatter).toSecondOfDay();
    }

    private static String trim(String value) {
        return value.replaceAll("[-/: .]", "");
    }

    @Override
    public String toString() {
        return type().name() + " " + name();
    }
}
