package io.intino.sumus.engine.filters;

import io.intino.sumus.engine.Dimension;
import io.intino.sumus.engine.Filter;
import io.intino.sumus.engine.Index;
import io.intino.sumus.engine.Slice;
import io.intino.sumus.engine.Slice.SliceIndex;

import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import static java.util.Arrays.asList;
import static java.util.stream.Collectors.*;

public class SliceFilter implements Filter {
	private final Set<Slice> slices;
	private final Set<Dimension> dimensions;
	private final Index index;

	public static Filter of(Slice... slices) {
		return of(asList(slices));
	}

	public static Filter of(List<Slice> slices) {
		if (slices.isEmpty()) return None;
		if (anyIsNull(slices)) return All;
		return new SliceFilter(slices);
	}

	private SliceFilter(List<Slice> slices) {
		this.slices = new HashSet<>(slices);
		this.dimensions = slices.stream().map(Slice::dimension).collect(toSet());
		this.index = new SliceIndex(this.slices);
	}

	@Override
	public boolean accepts(int idx) {
		return index.accepts(idx);
	}

	@Override
	public List<Slice> crop(List<Slice> slices) {
		return slices.stream().filter(this::accepts).collect(toList());
	}

	private boolean accepts(Slice slice) {
		return !dimensions.contains(slice.dimension()) || contains(slice);
	}

	private boolean contains(Slice slice) {
		return slice != null && (slices.contains(slice) || contains(slice.parent()));
	}

	@Override
	public String toString() {
		return slices.stream().map(Slice::name).collect(joining(" | "));
	}

	private static boolean anyIsNull(List<Slice> slices) {
		return slices.stream().anyMatch(Objects::isNull);
	}


}
