package io.intino.sumus.engine.builders;

import io.intino.sumus.engine.Cube;
import io.intino.sumus.engine.Fact;
import io.intino.sumus.engine.Filter;
import io.intino.sumus.engine.Slice;
import io.intino.sumus.engine.filters.SliceFilter;
import io.intino.sumus.model.IndicatorDefinition;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.*;

public class CellBuilder {
	public final List<IndicatorDefinition> indicators;
	public final Filter filter;
	public final List<Slice> slices;
	public final List<Accumulator> accumulators;

	public CellBuilder(List<IndicatorDefinition> indicators) {
		this.indicators = indicators;
		this.filter = Filter.None;
		this.slices = emptyList();
		this.accumulators = new ArrayList<>();
	}

	public CellBuilder(List<IndicatorDefinition> indicators, List<Slice> slices) {
		this.indicators = indicators;
		this.filter = SliceFilter.of(slices);
		this.slices = slices;
		this.accumulators = new ArrayList<>();
	}

	public void add(Accumulator accumulator) {
		this.accumulators.add(accumulator);
	}

	public void add(Fact fact) {
		boolean isAccepted = filter.accepts(fact.idx());
		for (Accumulator accumulator : accumulators) {
			accumulator.add();
			if (isAccepted) accumulator.add(fact.value(accumulator.name()));
		}

	}

	@Override
	public String toString() {
		return slices.stream().map(Slice::name).collect(joining("-"));
	}

	public Cube.Cell cell(Iterable<Fact> facts) {
		return new Cube.Cell() {
			@Override
			public List<Slice> slices() {
				return slices;
			}

			@Override
			public List<Cube.Indicator> indicators() {
				return translate(calculations());
			}

			private List<Cube.Indicator> translate(Map<String, Cube.Indicator> calculations) {
				return indicators.stream()
						.map(d -> indicator(calculations, d))
						.collect(toList());
			}

			// TODO: check
			private Cube.Indicator indicator(Map<String, Cube.Indicator> calculations, IndicatorDefinition d) {
				return calculations.getOrDefault(d.label(), Cube.Indicator.None).as(d.name(), d.unit(), ratio(d.scale()));
			}

			private double ratio(IndicatorDefinition.Scale scale) {
				return scale == null ? 1.0 : scale.ratio();
			}

			private Map<String, Cube.Indicator> calculations() {
				return accumulators.stream()
						.flatMap(a -> a.indicators().stream())
						.collect(toMap(Cube.Indicator::name, i -> i));
			}

			@Override
			public Iterable<Fact> facts() {
				return facts;
			}

			@Override
			public String toString() {
				return slices.stream()
						.map(Slice::toString)
						.collect(joining("-"));
			}
		};
	}

}
