package io.intino.sumus.chronos.timelines.blocks;

import io.intino.sumus.chronos.InvalidChronosBlockMarkException;
import io.intino.sumus.chronos.TimelineStore;
import io.intino.sumus.chronos.TimelineStore.TimeModel;

import java.nio.ByteBuffer;
import java.time.Instant;

import static java.nio.charset.StandardCharsets.UTF_8;

public class Header implements TimelineStore.Header {

	public static final int SIZE = 512;

	public String sensor;
	public long sensorModelPosition = 0;
	public long timeModelPosition = 0;
	public long recordCount = 0;
	public Instant first = Instant.EPOCH;
	public Instant last = Instant.EPOCH;
	public Instant next = Instant.EPOCH;

	public Header() {
	}

	public Header(String sensor) {
		this.setSensor(sensor);
	}

	@Override
	public String sensor() {
		return sensor;
	}

	@Override
	public long sensorModelPosition() {
		return sensorModelPosition;
	}

	@Override
	public long timeModelPosition() {
		return timeModelPosition;
	}

	@Override
	public long recordCount() {
		return recordCount;
	}

	@Override
	public Instant first() {
		return first == Instant.EPOCH ? null : first;
	}

	@Override
	public Instant last() {
		return last == Instant.EPOCH ? null : last;
	}

	@Override
	public Instant next() {
		return next == Instant.EPOCH ? null : next;
	}

	public void setSensor(String sensor) {
		if(sensor.getBytes(UTF_8).length > 255) throw new IllegalArgumentException("Sensor name length must be < 255 bytes (UTF-8)");
		this.sensor = sensor;
	}

	public void setTimeModel(long position, TimeModel timeModel) {
		setInstant(timeModel);
		this.timeModelPosition = position;
	}

	public void setInstant(TimeModel timeModel) {
		first = timeModel.instant();
		next = first;
		last = first;
	}

	public void step(TimeModel timeModel) {
		first = first == Instant.EPOCH ? next : first;
		last = next;
		next = timeModel.next(last);
		++recordCount;
	}

	public void setSensorModel(long position) {
		sensorModelPosition = position;
	}

	private boolean isFirstTimeModel() {
		return timeModelPosition == 0;
	}

	public static ByteBuffer serialize(TimelineStore.Header header) {
		return serialize(header, ByteBuffer.allocate(SIZE)).clear();
	}

	public static ByteBuffer serialize(TimelineStore.Header header, ByteBuffer buffer) {
		final int initialPosition = buffer.position();

		buffer.putShort(MARK);
		buffer.putLong(header.recordCount());
		buffer.putLong(header.sensorModelPosition());
		buffer.putLong(header.timeModelPosition());
		buffer.putLong(header.first().toEpochMilli());
		buffer.putLong(header.last().toEpochMilli());
		buffer.putLong(header.next().toEpochMilli());

		byte[] sensorUTF8 = header.sensor().getBytes(UTF_8);
		buffer.putShort((short) sensorUTF8.length);
		buffer.put(sensorUTF8);

		buffer.position(initialPosition + SIZE);
		return buffer;
	}

	public static TimelineStore.Header deserialize(ByteBuffer buffer) {
		return deserialize(buffer, new Header());
	}

	public static Header deserialize(ByteBuffer buffer, Header header) {
		short mark = buffer.getShort();
		if(mark != MARK) throw new InvalidChronosBlockMarkException("Header", MARK, mark);

		header.recordCount = buffer.getLong();
		header.sensorModelPosition = buffer.getLong();
		header.timeModelPosition = buffer.getLong();
		header.first = Instant.ofEpochMilli(buffer.getLong());
		header.last = Instant.ofEpochMilli(buffer.getLong());
		header.next = Instant.ofEpochMilli(buffer.getLong());

		short length = buffer.getShort();
		byte[] bytes = new byte[length];
		buffer.get(bytes);
		header.sensor = new String(bytes, UTF_8);

		return header;
	}
}
