package io.intino.sumus.chronos;

import io.intino.sumus.chronos.timelines.TimelineWriter;
import io.intino.sumus.chronos.timelines.stores.FileTimelineStore;
import io.intino.sumus.chronos.timelines.writers.FileTimelineWriter;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Iterator;

public interface TimelineStore {

	static TimelineStore of(File file) throws IOException {
		return new FileTimelineStore(file);
	}

	static TimelineStore.Builder createIfNotExists(String sensor, File file) {
		return new Builder(sensor, file);
	}

	default TimelineWriter writer() throws IOException {
		if(this instanceof FileTimelineStore)
			return new FileTimelineWriter(sensor(), ((FileTimelineStore)this).file(), true);
		throw new UnsupportedOperationException("No writers for " + getClass().getSimpleName());
	}

	Timeline timeline() throws IOException;

	Header header();
	SensorModel sensorModel();
	TimeModel timeModel();

	default String id() {return sensor();}
	default String sensor() {return header().sensor();}
	default Instant first() {return header().first();}
	default Instant last() {return header().last();}
	default Instant next() {return header().next();}

	default Period period() {
		if(timeModel() == null) throw new IllegalStateException("TimeModel is not set");
		return timeModel().period();
	}

	interface Block {
		short mark();
	}

	interface Header extends Block {

		short MARK = 0x5005;

		@Override
		default short mark() {
			return MARK;
		}

		String sensor();
		long sensorModelPosition();
		long timeModelPosition();
		long recordCount();
		Instant first();
		Instant last();
		Instant next();
	}

	interface TimeModel extends Block {

		short MARK = 0x6660;

		@Override
		default short mark() {
			return MARK;
		}

		Instant instant();
		Period period();
		ChronoUnit unit();
		long duration();
		Instant next(Instant instant);
	}

	interface SensorModel extends Block, Iterable<Magnitude> {

		short MARK = 0x6661;

		@Override
		default short mark() {
			return MARK;
		}

		Magnitude[] magnitudes();
		Magnitude get(int i);
		Magnitude get(String measurement);
		int indexOf(Magnitude magnitude);
		int indexOf(String magnitude);
		int size();
		boolean has(String measurement);
		boolean has(Magnitude magnitude);

		default boolean contains(SensorModel other) {
			Magnitude[] myMagnitudes = magnitudes();
			Magnitude[] otherMagnitudes = other.magnitudes();

			if(myMagnitudes.length < otherMagnitudes.length) return false;

			for(int i = 0;i < otherMagnitudes.length;i++)
				if(!myMagnitudes[i].equals(otherMagnitudes[i])) return false;

			return true;
		}

		@Override
		default Iterator<Magnitude> iterator() {
			return new Iterator<>() {
				int i = 0;

				@Override
				public boolean hasNext() {
					return i < size();
				}

				@Override
				public Magnitude next() {
					return get(i++);
				}
			};
		}
	}

	interface Data extends Block, Iterable<Data.Record> {

		short MARK = 0x5555;

		@Override
		default short mark() {
			return MARK;
		}

		int numRecords();
		int numMeasurementsPerRecord();
		int recordByteSize();
		ByteBuffer dataBuffer();
		Record record(int index);

		interface Record {
			int index();
			Instant instant();
			double[] values();
			double get(int index);
			int byteSize();
			int numMeasurements();
		}
	}

	class Builder {
		private final String sensor;
		private final File file;
		private TimeModel timeModel;
		private SensorModel sensorModel;

		public Builder(String sensor, File file) {
			this.sensor = sensor;
			this.file = file;
		}

		public TimelineStore build() throws IOException {
			if(file.exists()) return TimelineStore.of(file);
			file.getParentFile().mkdirs();
			try(TimelineWriter writer = new FileTimelineWriter(sensor, file, false)) {
				if(sensorModel != null) writer.sensorModel(sensorModel);
				if(timeModel != null) writer.timeModel(timeModel);
			}
			return TimelineStore.of(file);
		}

		public Builder withTimeModel(Instant start, Period period) {
			timeModel = new io.intino.sumus.chronos.timelines.blocks.TimeModel(start, period);
			return this;
		}

		public Builder withSensorModel(String... magnitudes) {
			sensorModel = new io.intino.sumus.chronos.timelines.blocks.SensorModel(magnitudes);
			return this;
		}

		public Builder withSensorModel(Magnitude... magnitudes) {
			sensorModel = new io.intino.sumus.chronos.timelines.blocks.SensorModel(magnitudes);
			return this;
		}
	}
}
