package io.intino.sumus.chronos;

import java.time.Instant;
import java.util.*;
import java.util.stream.Stream;

import static io.intino.sumus.chronos.Reel.State.Off;
import static io.intino.sumus.chronos.Reel.State.On;
import static java.lang.Integer.parseInt;
import static java.lang.Integer.toBinaryString;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.*;
import static java.util.stream.IntStream.range;

public class Reel {
	public final Instant from;
	public final Instant to;
	public final int step;
	public final int length;
	private final Map<String, byte[]> signals;
	private final List<Group> groups;

	private Reel(Instant from, Instant to, int step, int length, Map<String, byte[]> signals, List<Group> groups) {
		this.from = from;
		this.to = to;
		this.step = step;
		this.length = length;
		this.signals = signals;
		this.groups = groups;
	}

	public String get(String signal) {
		return new Line(signal).serialize();
	}

	public List<String> signals() {
		return new ArrayList<>(signals.keySet());
	}

	public List<String> signals(String group) {
		return signalsOf(find(group.hashCode()));
	}

	private List<String> signalsOf(Group group) {
		return group != null ? signals.keySet().stream().filter(group::contains).collect(toList()) : emptyList();
	}

	private Group find(String group) {
		return find(group.hashCode());
	}

	private Group find(int group) {
		return groups.stream().filter(g -> g.id == group).findFirst().orElse(new Group(group));
	}

	public Map<String, Shot> lastShots() {
		return signals.keySet().stream()
				.map(this::lastShot)
				.collect(toMap(s -> s.signal, s -> s));
	}

	public Map<String, Shot> lastShots(String group) {
		return lastShots(find(group));
	}

	private Map<String, Shot> lastShots(Group group) {
		return signals.keySet().stream()
				.filter(group::contains)
				.map(this::lastShot)
				.collect(toMap(s -> s.signal, s -> s));
	}

	public Shot lastShot(String signal) {
		return lastShot(new Line(signal));
	}

	private static Shot lastShot(Line line) {
		return new Shot(line.lastTransition(), State.of(line.last), line.signal);
	}

	public static class Shot {
		public final String signal;
		public final Instant ts;
		public final State state;
		public final int hash;

		public Shot(Instant ts, State state, int signal) {
			this.ts = ts;
			this.state = state;
			this.signal = "#" + signal;
			this.hash = signal;
		}

		public Shot(Instant ts, State state, String signal) {
			this.ts = ts;
			this.state = state;
			this.signal = signal;
			this.hash = signal.hashCode();
		}

		@Override
		public String toString() {
			return String.join(" ", ts.toString(), signal, state.name());
		}

		@Override
		public boolean equals(Object o) {
			if (o == null || getClass() != o.getClass()) return false;
			return this == o || equals((Shot) o);
		}

		private boolean equals(Shot shot) {
			return hash == shot.hash && Objects.equals(signal, shot.signal) && Objects.equals(ts, shot.ts) && state == shot.state;
		}

		@Override
		public int hashCode() {
			return Objects.hash(signal, ts, state, hash);
		}
	}

	public enum State {
		On, Off;

		public static State of(String value) {
			return of(value.equals("1"));
		}

		public static State of(boolean value) {
			return value ? On : Off;
		}
	}

	public static class Group {
		public final int id;
		public final Set<Integer> signals;

		public Group(int id) {
			this(id, new HashSet<>());
		}

		public Group(int id, Set<Integer> signals) {
			this.id = id;
			this.signals = signals;
		}

		public int id() {
			return id;
		}

		public State get(String signal) {
			return get(signal.hashCode());
		}

		public State get(int signal) {
			return State.of(contains(signal));
		}

		public void put(String signal) {
			put(signal.hashCode());
		}

		public void put(int signal) {
			if (contains(signal)) return;
			signals.add(signal);
		}

		public List<Shot> shotsToTurnOn(Instant now, String[] signals) {
			return Arrays.stream(signals)
					.filter(s -> !s.isEmpty() && !contains(s))
					.map(s -> new Shot(now, On, s))
					.collect(toList());
		}

		public List<Shot> shotsToTurnOff(Instant now, Set<Integer> signals) {
			return signals.stream()
					.map(s -> new Shot(now, Off, s))
					.collect(toList());
		}

		public Set<Integer> signalsThatAreNotIn(String[] signals) {
			Set<Integer> result = new HashSet<>(this.signals);
			result.removeAll(hashesOf(signals));
			return result;
		}

		public void remove(String signal) {
			remove(hashOf(signal));
		}

		public void remove(int signal) {
			if (!contains(signal)) return;
			signals.remove(signal);
		}

		public boolean contains(String signal) {
			return signals.contains(signal.hashCode());
		}

		public boolean contains(int signal) {
			return signals.contains(signal);
		}

		public static Set<Integer> hashesOf(String[] signals) {
			return Stream.of(signals).map(String::hashCode).collect(toSet());
		}

		public static int hashOf(String signal) {
			return signal.startsWith("#") ? parseInt(signal.substring(1)) : signal.hashCode();
		}

		@Override
		public String toString() {
			return id + "-" + signals.size();
		}
	}

	private class Line {
		final String signal;
		final byte[] bytes;
		final boolean last;

		Line(String signal) {
			this.signal = signal;
			this.bytes = signals.getOrDefault(signal, new byte[0]);
			this.last = get(length-1);
		}

		boolean get(int position) {
			int index = position / 8;
			return get(index < bytes.length ? bytes[index] : 0, position % 8);
		}

		boolean get(byte chunk, int bit) {
			return ((chunk >> (7-bit)) & 1) == 1;
		}

		Instant lastTransition() {
			return from.plusSeconds(findLastTransition() * step);
		}

		private long findLastTransition() {
			if (bytes.length == 0) return -1;

			for (int i = bytes.length - 1; i >= 0; i--) {
				byte chunk = bytes[i];
				if (chunk == (last ? -1 : 0)) continue;
				for (int bit = 7; bit >= 0; bit--)
					if (get(chunk, bit) != last)
						return i * 8L + bit + 1;
			}
			return -1;
		}

		private String serialize() {
			StringBuilder sb = new StringBuilder(length);
			for (byte b : bytes) sb.append(chunk(b));
			while (sb.length() < length) sb.append(' ');
			return sb.substring(0, length);
		}

	}

	public static class Builder {
		private final Instant from;
		private final Instant to;
		private final Map<Integer, String> names;
		private final Map<Integer, byte[]> signals;
		private final List<Group> groups;
		private int step;

		public Builder(Instant from, Instant to) {
			this(from, to, emptyList());
		}

		public Builder(Instant from, Instant to, List<Group> groups) {
			this.from = from;
			this.to = to;
			this.step = 1;
			this.names = new HashMap<>();
			this.signals = new HashMap<>();
			this.groups = groups;
		}

		public Builder add(Shot shot) {
			return shot.state == State.On ?
					on(shot.signal, shot.ts) :
					off(shot.hash, shot.ts);
		}

		public Builder on(String signal, Instant ts) {
			createSignalNameIfNotExists(signal.hashCode(), signal);
			return on(signal.hashCode(), ts);
		}

		private Builder on(int hash, Instant ts) {
			if (isInRange(ts))
				fillFrom(signalOf(hash), indexOf(ts), true);
			return this;
		}

		public Builder off(String signal, Instant ts) {
			return off(signal.hashCode(), ts);
		}

		public Builder off(int hash, Instant ts) {
			if (isInRange(ts)) fillFrom(signalOf(hash), indexOf(ts), false);
			return this;
		}

		private boolean isInRange(Instant ts) {
			return ts.getEpochSecond() <= to.getEpochSecond();
		}

		private byte[] signalOf(int hash) {
			createIfNotExists(hash);
			return signals.get(hash);
		}

		private void createIfNotExists(int hash) {
			if (signals.containsKey(hash)) return;
			signals.put(hash, new byte[size()]);
		}

		private void createSignalNameIfNotExists(int hash, String signal) {
			if (names.containsKey(hash)) return;
			names.put(hash, signal);
		}

		private int size() {
			return (int) Math.ceil(length() / 8.);
		}

		private int length() {
			return (int) indexOf(to);
		}

		private void fillFrom(byte[] values, long index, boolean set) {
			int position = (int) (index >> 3);
			if (position < 0 || position >= values.length) return;
			values[position] = set ?
					(byte) (values[position] | mask(offset(index))) :
					(byte) (values[position] & ~mask(offset(index)));
			byte value = (byte) (set ? 0xFF : 0);

			for (int i = position + 1, size = size(); i < size; i++) values[i] = value;
		}

		private static byte offset(long index) {
			return (byte) (index & 7);
		}

		private static byte mask(int offset) {
			return (byte) ((1 << (8 - offset)) - 1);
		}

		private long indexOf(Instant ts) {
			long index = (ts.getEpochSecond() - from.getEpochSecond()) / step;
			return index >= 0 ? index : 0;
		}

		public Reel close() {
			return new Reel(from, to, step, length(), consolidateSignals(), groups);
		}

		private Map<String, byte[]> consolidateSignals() {
			return signals.keySet().stream()
					.filter(this::hasAnyNonZero)
					.collect(toMap(names::get, signals::get));
		}

		private boolean hasAnyNonZero(int hash) {
			return hasAnyNonZero(signals.get(hash));
		}

		private static boolean hasAnyNonZero(byte[] bytes) {
			return range(0, bytes.length).anyMatch(i -> bytes[i] != 0);
		}

		public Builder by(Period period) {
			this.step = (int) period.duration();
			return this;
		}

		public Builder by(int seconds) {
			this.step = seconds;
			return this;
		}
	}


	private static final Map<Integer, String> Chunks = createChunks();

	private static String chunk(int i) {
		return Chunks.get(i & 0xFF);
	}

	private static Map<Integer, String> createChunks() {
		return range(0, 256).boxed()
				.collect(toMap(i -> i, Reel::createChunk));
	}

	private static String createChunk(int i) {
		return toBinaryString((i & 0xFF) + 0x100)
				.substring(1)
				.replace('0', ' ')
				.replace('1', '-');
	}
}
