package io.intino.sumus.chronos.timelines;

import io.intino.sumus.chronos.TimelineStore;
import io.intino.sumus.chronos.timelines.blocks.Data;
import io.intino.sumus.chronos.timelines.blocks.Header;
import io.intino.sumus.chronos.timelines.blocks.SensorModel;
import io.intino.sumus.chronos.timelines.blocks.TimeModel;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.SeekableByteChannel;
import java.time.Instant;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import static java.nio.file.StandardOpenOption.READ;
import static java.nio.file.StandardOpenOption.WRITE;

public class TimelineFixer implements AutoCloseable {

	private final SeekableByteChannel channel;
	private final TimelineStore.Header header;
	private final List<SensorModel> sensorModels;
	private final Map<Instant, Long> instantPositions;
	private TimelineStore.SensorModel sensorModel;

	public TimelineFixer(File file) throws IOException {
		this(FileChannel.open(file.toPath(), READ, WRITE));
	}

	public TimelineFixer(SeekableByteChannel channel) throws IOException {
		this.channel = channel;
		this.header = TimelineReader.readHeader(channel, new Header());
		this.sensorModels = new ArrayList<>();
		this.instantPositions = buildInstantPositionMap();
	}

	public TimelineFixer seek(Instant instant) throws IOException {
		if(!instantPositions.containsKey(instant)) throw new IllegalArgumentException("Instant " + instant + " not found");
		channel.position(instantPositions.get(instant));
		sensorModel = getCurrentSensorModel();
		return this;
	}

	public double get(String magnitude) throws IOException {
		int index = sensorModel.indexOf(magnitude);

		ByteBuffer buffer = ByteBuffer.allocate(Double.BYTES);
		long position = channel.position();
		channel.position(position + index * 8L);
		channel.read(buffer);
		channel.position(position);

		return buffer.getDouble(0);
	}

	public TimelineFixer set(String magnitude, double value) throws IOException {
		int index = sensorModel.indexOf(magnitude);

		ByteBuffer buffer = ByteBuffer.allocate(Double.BYTES);
		buffer.putDouble(0, value);

		long position = channel.position();
		channel.position(position + index * 8L);
		channel.write(buffer);
		channel.position(position);

		return this;
	}

	private TimelineStore.SensorModel getCurrentSensorModel() throws IOException {
		long currentPosition = channel.position();
		for(int i = sensorModels.size() - 1;i >= 0;i--) {
			SensorModel sensorModel = sensorModels.get(i);
			if(sensorModel.position() < currentPosition) return sensorModel;
		}
		return sensorModels.get(sensorModels.size() - 1);
	}

	private Map<Instant, Long> buildInstantPositionMap() throws IOException {
		channel.position(0);
		Map<Instant, Long> indices = new LinkedHashMap<>(Math.max((int) header.recordCount(), 16));

		TimelineReader reader = new TimelineReader(channel);
		TimelineStore.TimeModel timeModel = null;
		Instant[] instant = new Instant[1];

		while(reader.hasNext()) {
			TimelineStore.Block block = reader.next();
			switch (block.mark()) {
				case TimelineStore.Data.MARK:
					index((Data) block, timeModel, instant, indices);
					break;
				case TimelineStore.TimeModel.MARK:
					timeModel = (TimeModel) block;
					instant[0] = timeModel.instant();
					break;
				case TimelineStore.SensorModel.MARK:
					sensorModels.add((SensorModel) block);
					break;
			}
		}

		return indices;
	}

	private void index(Data block, TimelineStore.TimeModel timeModel, Instant[] instant, Map<Instant, Long> indices) {
		long position = block.position();
		if(position == -1) throw new IllegalStateException("Unknown data block position in the channel");
		position += Data.HEADER_SIZE;
		for(int i = 0;i < block.numRecords();i++) {
			indices.put(instant[0], position + ((long) i * block.recordByteSize()));
			instant[0] = timeModel.next(instant[0]);
		}
	}

	@Override
	public void close() throws IOException {
		channel.close();
	}

	@Override
	public String toString() {
		try {
			return "ChronosFixer{pos=" + channel.position() + "}";
		} catch (IOException e) {
			return getClass().getSimpleName();
		}
	}
}
