package io.intino.sumus.chronos;

import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import static java.lang.Integer.toBinaryString;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.IntStream.range;

public class Reel {
	public final Instant from;
	public final Instant to;
	public final int step;
	public final int length;
	private final Map<String, byte[]> signals;

	private Reel(Instant from, Instant to, int step, int length, Map<String, byte[]> signals) {
		this.from = from;
		this.to = to;
		this.step = step;
		this.length = length;
		this.signals = signals;
	}

	public Set<String> signals() {
		return signals.keySet();
	}

	public Map<String, Signal> last() {
		return signals.keySet().stream()
				.map(this::last)
				.collect(toMap(s->s.name, s->s));
	}

	public Signal last(String signal) {
		Line line = new Line(signal);
		return new Signal(signal, line.last, line.lastTransition());
	}

	private class Line {
		final byte[] bytes;
		final boolean last;

		Line(String signal) {
			this(signals.getOrDefault(signal, new byte[0]));
		}

		Line(byte[] bytes) {
			this.bytes = bytes;
			this.last = get(length-1);
		}

		boolean get(int position) {
			int index = position / 8;
			return get(index < bytes.length ? bytes[index] : 0, position % 8);
		}

		boolean get(byte chunk, int bit) {
			return ((chunk >> (7-bit)) & 1) == 1;
		}

		Instant lastTransition() {
			return from.plusSeconds(findLastTransition() * step);
		}

		private long findLastTransition() {
			if (bytes.length == 0) return -1;

			for (int i = bytes.length - 1; i >= 0; i--) {
				byte chunk = bytes[i];
				if (chunk == (last ? -1 : 0)) continue;
				for (int bit = 7; bit >= 0; bit--)
					if (get(chunk, bit) != last)
						return i * 8L + bit + 1;
			}
			return -1;
		}

	}

	public String get(String signal) {
		return serialize(signals.getOrDefault(signal, new byte[0]));
	}

	private String serialize(byte[] states) {
		StringBuilder sb = new StringBuilder(length);
		for (byte state : states) sb.append(chunk(state));
		while (sb.length() < length) sb.append(' ');
		return sb.substring(0, length);
	}

	private static final Map<Integer, String> Chunks = createChunks();

	private static String chunk(int i) {
		return Chunks.get(i & 0xFF);
	}

	private static Map<Integer, String> createChunks() {
		return range(0, 256).boxed()
				.collect(toMap(i -> i, Reel::createChunk));
	}

	private static String createChunk(int i) {
		return toBinaryString((i & 0xFF) + 0x100)
				.substring(1)
				.replace('0', ' ')
				.replace('1', '-');
	}

	public static class Signal {
		public final String name;
		public final boolean value;
		public final Instant instant;

		public Signal(String name, boolean value, Instant instant) {
			this.name = name;
			this.value = value;
			this.instant = instant;
		}

		@Override
		public String toString() {
			return name + '[' + (value ? '1' : '0') + ']' + " " + instant;
		}
	}

	public static class Builder {
		private final Instant from;
		private final Instant to;
		private final Map<Integer, String> names;
		private final Map<Integer, byte[]> signals;
		private int step;

		public Builder(Instant from, Instant to) {
			this.from = from;
			this.to = to;
			this.step = 1;
			this.names = new HashMap<>();
			this.signals = new HashMap<>();
		}

		public Builder add(Shot shot) {
			return shot.state == State.On ?
					on(shot.signal, shot.ts) :
					off(shot.hash, shot.ts);
		}

		public Builder on(String signal, Instant ts) {
			createSignalNameIfNotExists(signal.hashCode(), signal);
			return on(signal.hashCode(), ts);
		}

		private Builder on(int hash, Instant ts) {
			if (isInRange(ts))
				fillFrom(signalOf(hash), indexOf(ts), true);
			return this;
		}

		public Builder off(String signal, Instant ts) {
			return off(signal.hashCode(), ts);
		}

		public Builder off(int hash, Instant ts) {
			if (isInRange(ts)) fillFrom(signalOf(hash), indexOf(ts), false);
			return this;
		}

		private boolean isInRange(Instant ts) {
			return ts.getEpochSecond() <= to.getEpochSecond();
		}

		private byte[] signalOf(int hash) {
			createIfNotExists(hash);
			return signals.get(hash);
		}

		private void createIfNotExists(int hash) {
			if (signals.containsKey(hash)) return;
			signals.put(hash, new byte[size()]);
		}

		private void createSignalNameIfNotExists(int hash, String signal) {
			if (names.containsKey(hash)) return;
			names.put(hash, signal);
		}

		private int size() {
			return (int) Math.ceil(length() / 8.);
		}

		private int length() {
			return (int) indexOf(to);
		}

		private void fillFrom(byte[] values, long index, boolean set) {
			int position = (int) (index >> 3);
			if (position < 0 || position >= values.length) return;
			values[position] = set ?
					(byte) (values[position] | mask(offset(index))) :
					(byte) (values[position] & ~mask(offset(index)));
			byte value = (byte) (set ? 0xFF : 0);

			for (int i = position + 1, size = size(); i < size; i++) values[i] = value;
		}

		private static byte offset(long index) {
			return (byte) (index & 7);
		}

		private static byte mask(int offset) {
			return (byte) ((1 << (8 - offset)) - 1);
		}

		private long indexOf(Instant ts) {
			long index = (ts.getEpochSecond() - from.getEpochSecond()) / step;
			return index >= 0 ? index : 0;
		}

		public Reel close() {
			return new Reel(from, to, step, length(), consolidate());
		}

		private Map<String, byte[]> consolidate() {
			return signals.keySet().stream()
					.filter(this::hasAnyNonZero)
					.collect(toMap(names::get, signals::get));
		}

		private boolean hasAnyNonZero(int hash) {
			return hasAnyNonZero(signals.get(hash));
		}

		private static boolean hasAnyNonZero(byte[] bytes) {
			return range(0, bytes.length).anyMatch(i -> bytes[i] != 0);
		}

		public Builder by(Period period) {
			this.step = (int) period.duration();
			return this;
		}

		public Builder by(int seconds) {
			this.step = seconds;
			return this;
		}
	}


	public static class Shot {
		public final Instant ts;
		public final String signal;
		public final int hash;
		public final State state;

		public Shot(Instant ts, State state, int signal) {
			this.ts = ts;
			this.state = state;
			this.signal = "#" + signal;
			this.hash = signal;
		}

		public Shot(Instant ts, State state, String signal) {
			this.ts = ts;
			this.state = state;
			this.signal = signal;
			this.hash = signal.hashCode();
		}

		@Override
		public String toString() {
			return String.join(" ", ts.toString(), signal, state.toString());
		}
	}

	public enum State {
		On, Off;

		public static State of(String value) {
			return of(value.equals("1"));
		}

		public static State of(boolean value) {
			return value ? On : Off;
		}
	}

}
