package io.intino.sumus.chronos.models.descriptive.timeline;

import io.intino.sumus.chronos.Magnitude;
import io.intino.sumus.chronos.Timeline;
import io.intino.sumus.chronos.models.descriptive.sequence.Sequence;
import io.intino.sumus.chronos.models.descriptive.timeseries.Distribution;

import java.time.DayOfWeek;
import java.time.Instant;
import java.time.LocalDate;
import java.time.Month;
import java.time.temporal.ChronoField;
import java.util.*;
import java.util.stream.Stream;

import static java.time.ZoneOffset.UTC;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.IntStream.iterate;
import static java.util.stream.IntStream.range;

public class Summary implements Iterable<Summary.Point> {
	private final String[] symbols;
	private final Map<String, Integer> index;
	private final Map<Magnitude, Distribution[]> distributions;

	public static Builder of(Timeline timeline) {
		return new Builder(timeline);
	}

	private Summary(String[] symbols, Map<Magnitude, Distribution[]> distributions) {
		this.symbols = symbols;
		this.index = index(symbols);
		this.distributions = distributions;
	}

	private Map<String, Integer> index(String[] symbols) {
		Map<String, Integer> index = new HashMap<>();
		for (String symbol : symbols)
			index.put(symbol, index.size());
		return index;
	}

	public int length() {
		return symbols.length;
	}

	public String[] labels() {
		return symbols;
	}

	public int indexOf(String label) {
		return index.get(label);
	}

	public Set<Magnitude> measurements() {
		return distributions.keySet();
	}

	public Point first() {
		return point(0);
	}

	public Point last() {
		return point(length() - 1);
	}

	public Point point(String label) {
		return point(indexOf(label));
	}

	public Point point(int index) {
		return new Point(index);
	}

	public Point[] head(int length) {
		Point[] points = new Point[length];
		Arrays.setAll(points, this::point);
		return points;
	}

	public Point[] tail(int length) {
		Point[] points = new Point[length];
		Arrays.setAll(points, i -> point(length() - i - 1));
		return points;
	}

	public Stream<Point> stream() {
		return range(0, length()).mapToObj(this::point);
	}

	public Iterator<Point> iterator() {
		return new Iterator<>() {
			int i = 0;

			@Override
			public boolean hasNext() {
				return i < symbols.length;
			}

			@Override
			public Point next() {
				return point(i++);
			}
		};
	}

	public class Point {
		public final int index;

		private Point(int index) {
			assert index >= 0 : index < length();
			this.index = index;
		}

		public String label() {
			return symbols[index];
		}

		public Distribution distribution(String measurement) {
			return distribution(new Magnitude(measurement));
		}

		public Distribution distribution(Magnitude magnitude) {
			return distributions.get(magnitude)[index];
		}

		public Point next() {
			return point(index + 1);
		}

		public Point step(int offset) {
			return point(index + offset);
		}

		public Point prev() {
			return point(index - 1);
		}

		public Stream<Point> forward() {
			return range(index, length()).mapToObj(Summary.this::point);
		}

		public Stream<Point> backward() {
			return iterate(index, i -> i >= 0, i -> i - 1).mapToObj(Summary.this::point);
		}

		@Override
		public String toString() {
			return label();
		}

		@Override
		public boolean equals(Object o) {
			return this == o || o instanceof Point && index == ((Point) o).index;
		}

		@Override
		public int hashCode() {
			return Objects.hash(index);
		}

		private void add(Timeline.Point point) {
			for (Magnitude magnitude : point.magnitudes())
				distribution(magnitude).add(point.value(magnitude));
		}
	}


	public static class Builder {
		private final Timeline timeline;

		private Builder(Timeline timeline) {
			this.timeline = timeline;
		}

		public Summary by(Sequence.Quantization quantization) {
			Sequence sequence = new Sequence.Builder(timeline).by(quantization);
			Summary summary = new Summary(sequence.symbols(), distributions(sequence.symbols().length));
			for (Timeline.Point point : timeline)
				summary.point(quantization.get(point)).add(point);
			return summary;
		}

		private Map<Magnitude, Distribution[]> distributions(int size) {
			return timeline.magnitudes().stream().collect(toMap(m -> m, m -> distributions(m, size)));
		}

		private Distribution[] distributions(Magnitude magnitude, int length) {
			Distribution[] result = new Distribution[length];
			Arrays.setAll(result, i -> new Distribution(magnitude.model));
			return result;
		}

	}

	public static String byYear(Timeline.Point point) {
		return timetag(point.instant(), 4);
	}

	public static String byMonth(Timeline.Point point) {
		return timetag(point.instant(), 6);
	}

	public static String byWeek(Timeline.Point point) {
		LocalDate date = point.instant().atZone(UTC).toLocalDate();
		int year = date.get(ChronoField.YEAR);
		int week = date.get(ChronoField.ALIGNED_WEEK_OF_YEAR);
		return String.format("%dW%02d", year, week);
	}

	public static String days(Timeline.Point point) {
		return timetag(point.instant(), 8);
	}

	public static String hours(Timeline.Point point) {
		return timetag(point.instant(), 10);
	}

	public static String minutes(Timeline.Point point) {
		return timetag(point.instant(), 12);
	}

	public static String seconds(Timeline.Point point) {
		return timetag(point.instant(), 14);
	}

	public static String daysOfWeek(Timeline.Point point) {
		int index = point.instant().atZone(UTC).toLocalDate().get(ChronoField.DAY_OF_WEEK);
		return DayOfWeek.of(index).toString();
	}

	public static String monthsOfYear(Timeline.Point point) {
		int index = point.instant().atZone(UTC).toLocalDate().get(ChronoField.MONTH_OF_YEAR);
		return Month.of(index).toString();
	}

	public static String byHourOfDay(Timeline.Point point) {
		int index = point.instant().atZone(UTC).toLocalDate().get(ChronoField.HOUR_OF_DAY);
		return String.format("%02d", index);
	}

	private static String timetag(Instant instant, int size) {
		char[] result = new char[size];
		char[] chars = instant.toString().toCharArray();
		for (int i = 0, j = 0; j < size; i++) {
			char c = chars[i];
			if (c >= '0' && c <= '9') result[j++] = c;
		}
		return new String(result);
	}
}
