package io.intino.sumus.chronos;

import io.intino.sumus.chronos.Reel.Shot;
import io.intino.sumus.chronos.Reel.State;

import java.io.*;
import java.nio.file.Files;
import java.time.Instant;
import java.util.*;
import java.util.stream.Stream;

import static io.intino.sumus.chronos.Reel.State.Off;
import static io.intino.sumus.chronos.Reel.State.On;
import static java.lang.Integer.parseInt;
import static java.nio.file.StandardOpenOption.APPEND;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.*;

@SuppressWarnings("ResultOfMethodCallIgnored")
public class ReelFile {
	private final File file;
	private final Header header;
	private final Instant from;

	private ReelFile(File file, List<Group> groups) {
		this.file = file;
		this.header = new Header(groups);
		this.from = readFirstInstantFrom(file);
	}

	public static ReelFile open(File file) throws IOException {
		return new ReelFile(file, Header.read(file));
	}

	public static ReelFile create(File file) throws IOException {
		Header.create(file);
		return new ReelFile(file, emptyList());
	}

	public ReelFile set(Instant instant, String group, String... signals) throws IOException {
		return set(instant, header.get(group.hashCode()), signals);
	}

	public ReelFile set(Instant instant, Group group, String... signals) throws IOException {
		List<Shot> shots = new ArrayList<>();
		shots.addAll(group.shotsToTurnOff(instant, group.signalsThatAreNotIn(signals)));
		shots.addAll(group.shotsToTurnOn(instant, signals));
		return append(group, shots);
	}

	public Group groupOf(String signal) {
		return header.groups.values().stream().filter(g -> g.contains(signal)).findFirst().orElse(null);
	}

	public ReelFile append(String header, Shot... shots) throws IOException {
		return append(header, List.of(shots));
	}

	public ReelFile append(String group, List<Shot> shots) throws IOException {
		return append(header.get(group.hashCode()), shots);
	}

	private ReelFile append(Group group, List<Shot> shots) throws IOException {
		boolean dirty = false;
		header.add(group);
		for (Shot shot : shots) {
			if (group.get(hashOf(shot.signal)) == shot.state) continue;
			Files.write(file.toPath(), serialize(shot), APPEND);
			if (shot.state == On)
				group.put(shot.signal);
			else
				group.remove(shot.signal);
			dirty = true;
		}
		if (dirty) header.write(file);
		return this;
	}

	private static Set<Integer> hashesOf(String[] signals) {
		return Stream.of(signals).map(String::hashCode).collect(toSet());
	}

	public Instant start() {
		return from;
	}

	private static int hashOf(String signal) {
		return signal.startsWith("#") ? parseInt(signal.substring(1)) : signal.hashCode();
	}

	private byte[] serialize(Shot shot) {
		try (ByteArrayOutputStream bos = new ByteArrayOutputStream();
			 DataOutputStream dos = new DataOutputStream(bos)) {
			dos.writeLong(shot.ts.getEpochSecond());
			dos.writeByte(shot.state == On ? 1 : 0);
			if (shot.state == On)
				dos.writeUTF(shot.signal);
			else
				dos.writeInt(shot.hash);
			return bos.toByteArray();
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	public Processor reel(Instant from, Instant to) {
		return new Processor(from, to);
	}

	public Processor reel(Instant to) {
		return new Processor(from, to);
	}

	public Processor reel() {
		return new Processor(from, Instant.now());
	}

	public State stateOf(String signal) {
		return State.of(header.contains(signal));
	}

	private Instant readFirstInstantFrom(File file) {
		try (DataInputStream is = new DataInputStream(new BufferedInputStream(new FileInputStream(file)))) {
			is.skip(Header.SIZE);
			long epochSecond = is.readLong();
			return Instant.ofEpochSecond(epochSecond);
		} catch (IOException e) {
			return Instant.now();
		}
	}


	public static class Group {
		private final int id;
		private final Set<Integer> signals;

		public Group(int id) {
			this(id, new HashSet<>());
		}

		public Group(int id, Set<Integer> signals) {
			this.id = id;
			this.signals = signals;
		}

		public int id() {
			return id;
		}

		public State get(String signal) {
			return get(signal.hashCode());
		}

		public State get(int signal) {
			return State.of(contains(signal));
		}

		public void put(String signal) {
			put(signal.hashCode());
		}

		public void put(int signal) {
			if (contains(signal)) return;
			signals.add(signal);
		}

		public List<Shot> shotsToTurnOn(Instant now, String[] signals) {
			return Arrays.stream(signals)
					.filter(s -> !s.isEmpty() && !contains(s))
					.map(s -> new Shot(now, On, s))
					.collect(toList());
		}

		private List<Shot> shotsToTurnOff(Instant now, Set<Integer> signals) {
			return signals.stream()
					.map(s -> new Shot(now, Off, s))
					.collect(toList());
		}

		private Set<Integer> signalsThatAreNotIn(String[] signals) {
			Set<Integer> result = new HashSet<>(this.signals);
			result.removeAll(hashesOf(signals));
			return result;
		}

		public void remove(String signal) {
			remove(hashOf(signal));
		}

		public void remove(int signal) {
			if (!contains(signal)) return;
			signals.remove(signal);
		}

		public boolean contains(String signal) {
			return signals.contains(signal.hashCode());
		}

		public boolean contains(int signal) {
			return signals.contains(signal);
		}

		@Override
		public String toString() {
			return id + "-" + signals.size();
		}
	}

	public class Processor {
		private final Instant from;
		private final Instant to;

		public Processor(Instant from, Instant to) {
			this.from = from;
			this.to = to;
		}

		public Reel by(Period period) {
			Reel.Builder builder = new Reel.Builder(from, to).by(period);

			try (DataInputStream is = new DataInputStream(new BufferedInputStream(new FileInputStream(file)))) {
				is.skipBytes(Header.SIZE);
				while (true) {
					long seconds = is.readLong();
					byte state = is.readByte();
					if (state == 1) {
						String signal = is.readUTF();
						builder.add(on(seconds, signal));
					} else {
						int signal = is.readInt();
						builder.add(off(seconds, signal));
					}
				}
			} catch (IOException e) {
				return builder.close();
			}
		}

		private Shot on(long seconds, String signal) {
			return new Shot(Instant.ofEpochSecond(seconds), On, signal);
		}

		private Shot off(long seconds, int signal) {
			return new Shot(Instant.ofEpochSecond(seconds), Off, signal);
		}
	}

	private static class Header {
		private static final int SIZE = 4096;
		private final Map<Integer, Group> groups;

		public Header(List<Group> groups) {
			this.groups = groups.stream().collect(toMap(h -> h.id, h -> h));
		}

		public void add(Group group) {
			if (groups.containsKey(group.id)) return;
			groups.put(group.id, group);
		}

		public boolean contains(String signal) {
			return groups.values().stream().anyMatch(g -> g.contains(signal));
		}

		public Group get(int id) {
			return groups.getOrDefault(id, new Group(id));
		}

		public static void create(File file) throws IOException {
			try (RandomAccessFile raf = new RandomAccessFile(file, "rw")) {
				raf.write(new byte[SIZE]);
			}
		}

		public static List<Group> read(File file) {
			List<Group> list = new ArrayList<>();
			try (DataInputStream is = new DataInputStream(new FileInputStream(file))) {
				while (true) {
					int id = is.readInt();
					if (id == 0) break;
					Group group = read(id, is);
					list.add(group);
				}
			} catch (IOException e) {
				throw new RuntimeException(e);
			}
			return list;
		}

		private static Group read(int id, DataInputStream is) throws IOException {
			Set<Integer> set = new HashSet<>();
			while (true) {
				int signal = is.readInt();
				if (signal == 0) break;
				set.add(signal);
			}
			return new Group(id, set);
		}

		public void write(File file) throws IOException {
			try (RandomAccessFile raf = new RandomAccessFile(file, "rw")) {
				raf.seek(0);
				for (int id : groups.keySet()) {
					raf.writeInt(id);
					for (int signal : signalsOf(id))
						raf.writeInt(signal);
					raf.writeInt(0);
				}
				raf.writeInt(0);
			}
		}

		private Set<Integer> signalsOf(int name) {
			return get(name).signals;
		}

	}
}
