package io.intino.sumus.chronos;

import io.intino.sumus.chronos.Reel.Shot;
import io.intino.sumus.chronos.Reel.State;

import java.io.*;
import java.nio.file.Files;
import java.time.Instant;
import java.util.*;
import java.util.stream.Stream;

import static io.intino.sumus.chronos.Reel.State.Off;
import static io.intino.sumus.chronos.Reel.State.On;
import static java.lang.Integer.parseInt;
import static java.nio.file.StandardOpenOption.APPEND;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

@SuppressWarnings("ResultOfMethodCallIgnored")
public class ReelFile {
	private final File file;
	private final Header header;
	private final Instant from;

	private ReelFile(File file) throws IOException {
		this.file = file;
		this.header = new Header();
		this.from = readFirstInstantFrom(file);
	}

	public static ReelFile open(File file) throws IOException {
		return new ReelFile(file);
	}

	public ReelFile set(Instant instant, String... signals) throws IOException {
		List<Shot> shots = new ArrayList<>();
		shots.addAll(shotsToTurnOff(instant, signalsThatAreNotIn(signals)));
		shots.addAll(shotsToTurnOn(instant, signals));
		return append(shots);
	}

	private List<Shot> shotsToTurnOn(Instant now, String[] signals) {
		return Arrays.stream(signals)
				.filter(s-> !header.contains(s))
				.map(s -> new Shot(now, On, s))
				.collect(toList());
	}

	private List<Shot> shotsToTurnOff(Instant now, Set<Integer> signals) {
		return signals.stream()
				.map(s->new Shot(now, Off, s))
				.collect(toList());
	}

	private Set<Integer> signalsThatAreNotIn(String[] signals) {
		Set<Integer> result = new HashSet<>(header.signals);
		result.removeAll(hashesOf(signals));
		return result;
	}

	private Set<Integer> hashesOf(String[] signals) {
		return Stream.of(signals).map(String::hashCode).collect(toSet());
	}

	public Instant start() {
		return from;
	}

	public ReelFile append(Shot... shots) throws IOException {
		return append(List.of(shots));
	}

	public ReelFile append(List<Shot> shots) throws IOException {
		boolean dirty = false;
		for (Shot shot : shots) {
			if (stateOf(shot.signal) == shot.state) continue;
			Files.write(file.toPath(), serialize(shot), APPEND);
			updateHeaderWith(shot);
			dirty = true;
		}
		if (dirty) header.write();
		return this;
	}



	public State stateOf(String signal) {
		return header.get(hashOf(signal));
	}

	private int hashOf(String signal) {
		return signal.startsWith("#") ? parseInt(signal.substring(1)) : signal.hashCode();
	}

	private void updateHeaderWith(Shot shot) {
		if (shot.state == On)
			header.put(shot.signal);
		else
			header.remove(shot.signal);
	}

	private byte[] serialize(Shot shot) {
		try (ByteArrayOutputStream bos = new ByteArrayOutputStream();
			 DataOutputStream dos = new DataOutputStream(bos)) {
			dos.writeLong(shot.ts.getEpochSecond());
			dos.writeByte(shot.state == On ? 1 : 0);
			if (shot.state == On)
				dos.writeUTF(shot.signal);
			else
				dos.writeInt(shot.hash);
			return bos.toByteArray();
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	public Processor reel(Instant from, Instant to) {
		return new Processor(from, to);
	}

	public Processor reel(Instant to) {
		return new Processor(from, to);
	}

	public Processor reel() {
		return new Processor(from, Instant.now());
	}

	private class Header {
		private static final int SIZE = 2048;
		private final Set<Integer> signals;

		public Header() throws IOException {
			this.signals = file.exists() ? read() : create();
		}

		private Set<Integer> read() {
			Set<Integer> set = new HashSet<>();
			try (DataInputStream is = new DataInputStream(new FileInputStream(file))) {
				final int max = SIZE >> 2;
				while (set.size() < max) {
					int signal = is.readInt();
					if (signal == 0) break;
					set.add(signal);
				}
			} catch (IOException e) {
				throw new RuntimeException(e);
			}
			return set;
		}

		private Set<Integer> create() throws IOException {
			try (RandomAccessFile raf = new RandomAccessFile(file, "rw")) {
				raf.write(new byte[SIZE]);
			}
			return new HashSet<>();
		}

		public void write() throws IOException {
			try (RandomAccessFile raf = new RandomAccessFile(file, "rw")) {
				raf.seek(0);
				for (int signal : signals) raf.writeInt(signal);
				raf.write(0);
			}
		}

		public State get(String signal) {
			return get(signal.hashCode());
		}

		public State get(int signal) {
			return State.of(contains(signal));
		}

		public void put(String signal) {
			put(signal.hashCode());
		}

		public void put(int signal) {
			if (contains(signal)) return;
			signals.add(signal);
		}

		public void remove(String signal) {
			remove(hashOf(signal));
		}

		public void remove(int signal) {
			if (!contains(signal)) return;
			signals.remove(signal);
		}

		public boolean contains(String signal) {
			return signals.contains(signal.hashCode());
		}


		public boolean contains(int signal) {
			return signals.contains(signal);
		}

	}

	private Instant readFirstInstantFrom(File file) {
		try (DataInputStream is = new DataInputStream(new BufferedInputStream(new FileInputStream(file)));) {
			is.skip(Header.SIZE);
			return Instant.ofEpochSecond(is.readLong());
		} catch (IOException e) {
			return Instant.now();
		}
	}


	public class Processor {
		private final Instant from;
		private final Instant to;

		public Processor(Instant from, Instant to) {
			this.from = from;
			this.to = to;
		}

		public Reel by(Period period) {
			Reel.Builder builder = new Reel.Builder(from, to).by(period);

			try (DataInputStream is = new DataInputStream(new BufferedInputStream(new FileInputStream(file)));) {
				is.skip(Header.SIZE);
				while (true) {
					long seconds = is.readLong();
					byte state = is.readByte();
					if (state == 1)
						builder.add(on(seconds, is.readUTF()));
					else
						builder.add(off(seconds, is.readInt()));
				}
			} catch (IOException e) {
				return builder.close();
			}
		}

		private Shot on(long seconds, String signal) {
			return new Shot(Instant.ofEpochSecond(seconds), On, signal);
		}

		private Shot off(long seconds, int signal) {
			return new Shot(Instant.ofEpochSecond(seconds), Off, signal);
		}



	}
}
