package io.intino.sumus.time.processors;

import io.intino.sumus.time.Magnitude;
import io.intino.sumus.time.Period;
import io.intino.sumus.time.Timeline;

import java.time.Instant;
import java.util.Iterator;
import java.util.Set;
import java.util.stream.DoubleStream;

public class Resampler {
	private final Timeline timeline;
	private final Instant[] instants;
	private final Set<Magnitude> magnitudes;

	public Resampler(Timeline timeline) {
		this.timeline = timeline;
		this.instants = timeline.instants;
		this.magnitudes = timeline.magnitudes();
	}

	public Timeline execute(Period period, int size) {
		Timeline.Builder builder = new Timeline.Builder(size);
		Iterator<Instant> iterator = period.iterator(timeline.first().instant());
		Position from = new Position(iterator.next(), 0);
		while (!builder.isComplete()) {
			Position to = new Position(iterator.next(), from.index);
			execute(builder, from, to);
			from = to;
		}
		return builder.close();
	}

	private void execute(Timeline.Builder builder, Position from, Position to) {
		builder.set(from.instant);
		for (Magnitude magnitude : magnitudes) {
			builder.set(magnitude, reduce(magnitude, from, to));
		}
	}

	private double reduce(Magnitude magnitude, Position from, Position to) {
		return magnitude.reduce(streamOf(magnitude, from, to));
	}

	private DoubleStream streamOf(Magnitude magnitude, Position from, Position to) {
		Timeline.Point point = startingOn(from.instant);
		return point == null ? DoubleStream.empty() : point.forward()
				.limit(to.index - from.index)
				.mapToDouble(p -> p.value(magnitude));
	}

	private Timeline.Point startingOn(Instant instant) {
		return instant.isBefore(timeline.instants[0]) ?
				timeline.first() :
				timeline.at(instant);
	}

	private class Position {
		final Instant instant;
		final int index;

		Position(Instant instant, int index) {
			this.instant = instant;
			this.index = instant == null ? instants.length : indexOf(instant, index);
		}

		private int indexOf(Instant instant, int index) {
			int i = index;
			while (i < timeline.length() && instants[i].isBefore(instant)) i++;
			return i;
		}

		@Override
		public String toString() {
			return instant != null ? instant.toString() : "EOT";
		}

	}

}
