package io.intino.sumus.time;

import io.intino.sumus.time.filters.Denoise;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

public interface Program {
	Timeline run(Timeline timeline);

	class Manual implements Program {
		private final List<Sentence> sentences;

		public Manual() {
			this.sentences = new ArrayList<>();
		}

		public Manual add(Sentence sentence) {
			sentences.add(sentence);
			return this;
		}

		@Override
		public Timeline run(Timeline timeline) {
			return new Execution(timeline).execute();
		}

		private class Execution {
			private final Timeline input;
			private final Timeline.Builder builder;

			Execution(Timeline timeline) {
				this.input = timeline;
				this.builder = new Timeline.Builder(input.instants);
			}

			Timeline execute() {
				sentences.forEach(sentence -> sentence.execute(input, builder));
				return builder.close();
			}

		}

		public interface Instruction {

		}

		public interface Method extends Instruction {
			Timeline execute(Timeline timeline);
		}

		public interface Operator extends Instruction {
			TimeSeries execute(TimeSeries timeSeries);
		}

		public static class Sentence {
			public final Instruction instruction;
			private String source;
			private String target;

			private Sentence(Instruction instruction) {
				this.instruction = instruction;
			}

			public static Sentence calculate(Operator operator) {
				return new Sentence(operator);
			}

			public static Sentence include(Method method) {
				return new Sentence(method);
			}

			public void execute(Timeline input, Timeline.Builder builder) {
				if (instruction instanceof Operator) execute(builder, (Operator) instruction);
				if (instruction instanceof Method) execute(input, builder, (Method) instruction);
			}

			private void execute(Timeline timeline, Timeline.Builder builder, Method method) {
				Timeline result = method.execute(timeline);
				builder.put(result);
			}

			private void execute(Timeline.Builder builder, Operator operator) {
				Magnitude source = builder.get(this.source);
				Magnitude target = new Magnitude(this.target, source.model);
				TimeSeries result = operator.execute(builder.series(source));
				builder.put(target, result);
			}

			public Sentence of(String source) {
				this.source = source;
				return this;
			}

			public Sentence as(String target) {
				this.target = target;
				return this;
			}
		}

		public static Operator inverse() {
			return TimeSeries::inverse;
		}

		public static Operator negate() {
			return TimeSeries::negate;
		}

		public static Operator square() {
			return TimeSeries::square;
		}

		public static Operator differential() {
			return TimeSeries::differential;
		}

		public static Operator rateOfGrowth() {
			return TimeSeries::rateOfGrowth;
		}

		public static Operator log() {
			return TimeSeries::log;
		}

		public static Operator movingAverage(int observations) {
			return t -> t.movingAverage(observations);
		}

		public static Operator movingAverage(double factor) {
			return t -> t.movingAverage(factor);
		}

		public static Operator denoise(Denoise.Mode mode) {
			return t -> t.denoise(mode);
		}

		public static Operator normalize() {
			return TimeSeries::normalize;
		}

		public static Operator normalize(double min, double max) {
			return t -> t.normalize(min, max);
		}

		public static Operator standardize() {
			return TimeSeries::standardize;
		}

		public static Method all() {
			return timeline -> timeline;
		}

		public static Method total(Function<Magnitude, Magnitude> function) {
			return timeline -> timeline.compose(function);
		}

		public static Method total() {
			return timeline -> timeline.compose(m->new Magnitude("total"));
		}


	}
}
