package io.intino.sumus.time;

import com.github.luben.zstd.Zstd;

import java.io.*;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.function.Consumer;

import static java.lang.Double.NaN;
import static java.lang.Double.isNaN;
import static java.lang.System.arraycopy;
import static java.util.Arrays.copyOfRange;
import static java.util.Arrays.stream;
import static java.util.stream.Collectors.joining;

public class ChronosFile {
	private final File file;
	private final Header header;
	private TimeModel timeModel;
	private SensorModel sensorModel;

	private ChronosFile(File file, String sensor) throws IOException {
		this.file = file;
		this.header = new Header(sensor);
		this.header.commit();
	}

	private ChronosFile(File file) throws IOException {
		this.file = file;
		this.header = new Header();
		if (file.exists()) loadModels();
	}

	public static ChronosFile create(String filename, String sensor) throws IOException {
		return create(new File(filename), sensor);
	}

	public static ChronosFile create(File file, String sensor) throws IOException {
		if (file.exists()) throw new IOException("File already exists");
		return new ChronosFile(file, sensor);
	}

	public static ChronosFile open(String filename) throws IOException {
		return open(new File(filename));
	}

	public static ChronosFile open(File file) throws IOException {
		if (!file.exists()) throw new IOException("File does not exist");
		return new ChronosFile(file);
	}

	public String id() {
		return header.sensor;
	}

	public int count() {
		return (int) header.count;
	}

	public Instant first() {
		return header.first;
	}

	public Instant last() {
		return header.last;
	}

	public Instant next() {
		return header.next;
	}

	public TimeModel timeModel() {
		return timeModel;
	}

	public SensorModel sensorModel() {
		return sensorModel;
	}

	public boolean compressed() {
		return header.compressed;
	}

	public ChronosFile compressed(boolean value) throws IOException {
		if (header.count > 0) throw new IllegalCallerException("Can not set compression of a file with data");
		header.compressed = value;
		header.commit();
		return this;
	}

	public Timeline timeline() throws IOException {
		Query query = new Query();
		read(query);
		return query.close();
	}

	public void export(File file) throws IOException {
		try (ItlExporter exporter = new ItlExporter(file)) {
			read(exporter);
		}
	}

	public ChronosFile timeModel(Instant instant, Period period) throws IOException {
		timeModel = new TimeModel(instant, period);
		updateHeader(instant);
		timeModel.commit();
		return this;
	}

	public ChronosFile sensorModel(String... measurements) throws IOException {
		sensorModel = new SensorModel(measurements);
		updateHeader();
		sensorModel.commit();
		return this;
	}

	public ChronosFile sensorModel(Magnitude... magnitudes) throws IOException {
		sensorModel = new SensorModel(magnitudes);
		updateHeader();
		sensorModel.commit();
		return this;
	}

	private void updateHeader(Instant instant) throws IOException {
		long position = file.length();
		header.setTimeModel(position, instant).commit();
	}

	private void updateHeader() throws IOException {
		header.setSensorModel(file.length()).commit();
	}

	public ChronosFile add(Timeline timeline) throws IOException {
		assert last().getEpochSecond() <= timeline.first().instant().getEpochSecond();
		DataSession session = add();
		add(timeline, session);
		session.close();
		return this;
	}

	public DataSession add() throws IOException {
		if (timeModel == null) throw new IllegalCallerException("It is not possible to add data without a time model");
		if (sensorModel == null) throw new IllegalCallerException("It is not possible to add data without a sensor model");
		return new DataSession(outputStream());
	}

	private void add(Timeline timeline, DataSession session) throws IOException {
		for (Timeline.Point point : timeline) {
			session.set(point.instant());
			for (Magnitude magnitude : sensorModel)
				session.set(magnitude, point.value(magnitude));
		}
	}

	private void loadModels() throws IOException {
		try (RandomAccessFile raf = new RandomAccessFile(file, "r")) {
			if (timeModelExists()) this.timeModel = loadTimeModel(raf);
			if (sensorModelExists()) this.sensorModel = loadSensorModel(raf);
		}
	}

	private boolean timeModelExists() {
		return header.timeModelPosition != 0;
	}

	private boolean sensorModelExists() {
		return header.sensorModelPosition != 0;
	}

	private Instant next(Instant instant) {
		return timeModel.next(instant);
	}

	private TimeModel loadTimeModel(RandomAccessFile raf) throws IOException {
		raf.seek(header.timeModelPosition);
		if (raf.readShort() != TimeModel.MARK) throw new RuntimeException("Corrupted file at loading time model");
		return new TimeModel(raf);
	}

	private SensorModel loadSensorModel(RandomAccessFile raf) throws IOException {
		raf.seek(header.sensorModelPosition);
		if (raf.readShort() != SensorModel.MARK) throw new RuntimeException("Corrupted file at loading sensor model");
		return new SensorModel(raf);
	}

	DataInputStream inputStream() throws IOException {
		return new DataInputStream(new BufferedInputStream(new FileInputStream(file)));
	}

	DataOutputStream outputStream() throws IOException {
		return new DataOutputStream(new BufferedOutputStream(new FileOutputStream(file, true)));
	}

	private void read(Consumer<Block> consumer) throws IOException {
		new Reader(consumer).read();
	}

	public Fix fix(Instant instant) throws IOException {
		return new Fix(instant);
	}

	public Period period() {
		return timeModel.period;
	}

	private class Header {
		private static final short MARK = 0x5005;
		private static final int SIZE = 512;
		private final String sensor;
		private boolean compressed;
		private long sensorModelPosition;
		private long timeModelPosition;
		private long count;
		private Instant first;
		private Instant last;
		private Instant next;

		private Header(String sensor) {
			this.sensor = sensor;
			this.sensorModelPosition = 0;
			this.timeModelPosition = 0;
			this.count = 0;
			this.compressed = false;
			this.first = Instant.EPOCH;
			this.next = Instant.EPOCH;
			this.last = Instant.EPOCH;
		}

		private Header() throws IOException {
			try (DataInputStream is = inputStream()) {
				if (is.readShort() != MARK) throw new RuntimeException("Corrupted file");
				this.count = is.readLong();
				this.sensorModelPosition = is.readLong();
				this.timeModelPosition = is.readLong();
				this.first = Instant.ofEpochMilli(is.readLong());
				this.last = Instant.ofEpochMilli(is.readLong());
				this.next = Instant.ofEpochMilli(is.readLong());
				this.compressed = is.readBoolean();
				this.sensor = is.readUTF();
			}
		}

		private void commit() throws IOException {
			boolean fileExists = file.exists();
			try (RandomAccessFile raf = new RandomAccessFile(file, "rw")) {
				raf.seek(0);
				raf.writeShort(MARK);
				raf.writeLong(count);
				raf.writeLong(sensorModelPosition);
				raf.writeLong(timeModelPosition);
				raf.writeLong(first.toEpochMilli());
				raf.writeLong(last.toEpochMilli());
				raf.writeLong(next.toEpochMilli());
				raf.writeBoolean(compressed);
				if (fileExists) return;
				raf.writeUTF(sensor);
				raf.write(fill(raf));
			}
		}

		private byte[] fill(RandomAccessFile raf) throws IOException {
			return new byte[(int) (SIZE - raf.length())];
		}

		Header setTimeModel(long position, Instant last) {
			this.first = isFirstTimeModel() ? last : this.first;
			this.last = last;
			this.next = timeModel().next(last);
			this.timeModelPosition = position;
			return this;
		}

		Header setSensorModel(long position) {
			sensorModelPosition = position;
			return this;
		}

		void step() {
			last = next;
			next = timeModel.next(last);
		}

		private boolean isFirstTimeModel() {
			return timeModelPosition == 0;
		}
	}

	private interface Block {
		Short NONE = (short) -1;
	}

	private interface BlockReader {
		void process(DataInputStream is) throws IOException;
	}

	public class TimeModel implements Block {
		static final short MARK = 0x6660;
		public final Instant instant;
		public final Period period;

		private TimeModel(Instant instant, Period period) {
			this.instant = instant;
			this.period = period;
		}

		private TimeModel(RandomAccessFile raf) throws IOException {
			this(raf.readLong(), raf.readShort(), raf.readShort());
		}

		private TimeModel(long instant, short amount, short chronoUnit) {
			this(Instant.ofEpochMilli(instant), Period.each(amount, chronoUnit));
		}

		private void commit() throws IOException {
			try (DataOutputStream os = outputStream()) {
				writeIn(os);
			}
		}

		private void writeIn(DataOutputStream output) throws IOException {
			output.writeShort(MARK);
			output.writeLong(instant.toEpochMilli());
			output.writeShort(period.amount);
			output.writeShort(period.unit.ordinal());
		}

		public ChronoUnit unit() {
			return period.unit;
		}

		public long duration() {
			return period.duration();
		}

		public Instant next(Instant instant) {
			return period.next(instant);
		}

		@Override
		public String toString() {
			return "from " + instant + " each " + period;
		}
	}

	public class SensorModel implements Block, Iterable<Magnitude> {
		static final short MARK = 0x6661;
		public final Magnitude[] magnitudes;
		public final Map<Magnitude, Integer> index = new HashMap<>();

		private SensorModel(String... measurements) {
			this(stream(measurements).map(Magnitude::new).toArray(Magnitude[]::new));
		}

		private SensorModel(Magnitude... magnitudes) {
			this.magnitudes = magnitudes;
			this.index(magnitudes);
		}

		private SensorModel(RandomAccessFile raf) throws IOException {
			this(readBigString(raf).split("\n"));
		}

		public Magnitude get(int i) {
			return i >= 0 ? magnitudes[i] : null;
		}

		public Magnitude get(String measurement) {
			return get(indexOf(new Magnitude(measurement)));
		}

		public int indexOf(Magnitude magnitude) {
			return index.getOrDefault(magnitude,-1);
		}

		public int size() {
			return magnitudes.length;
		}

		public boolean has(String measurement) {
			return index.containsKey(new Magnitude(measurement));
		}

		public boolean has(Magnitude magnitude) {
			return index.containsKey(magnitude);
		}

		private void index(Magnitude[] magnitudes) {
			for (int i = 0; i < magnitudes.length; i++)
				index.put(magnitudes[i], i);
		}

		private void commit() throws IOException {
			try (DataOutputStream os = outputStream()) {
				writeIn(os);
			}
		}

		private void writeIn(DataOutputStream output) throws IOException {
			output.writeShort(MARK);
			writeBigString(output, String.join("\n", magnitudes));
		}


		@Override
		public Iterator<Magnitude> iterator() {
			return new Iterator<>() {
				int i = 0;

				@Override
				public boolean hasNext() {
					return i < size();
				}

				@Override
				public Magnitude next() {
					return magnitudes[i++];
				}
			};
		}

		@Override
		public boolean equals(Object o) {
			if (o == null || getClass() != o.getClass()) return false;
			return this == o || Arrays.equals(magnitudes, ((SensorModel) o).magnitudes);
		}

		@Override
		public int hashCode() {
			return Arrays.hashCode(magnitudes);
		}

		@Override
		public String toString() {
			return stream(magnitudes).map(Magnitude::toString).collect(joining("\n"));
		}
	}

	public static class Data implements Block {
		static final short SIZE = 6;
		static final short MARK = 0x5555;
		public final Instant instant;
		public final double[] values;

		private Data(Instant instant, double[] values) {
			this.instant = instant;
			this.values = values;
		}

		public Instant instant() {
			return instant;
		}

		public double get(int index) {
			return values[index];
		}

		public byte[] serialize() {
			try (ByteArrayOutputStream data = new ByteArrayOutputStream(); DataOutputStream os = new DataOutputStream(data)) {
				for (double value : values) os.writeDouble(value);
				return data.toByteArray();
			} catch (IOException ignored) {
				return new byte[0];
			}
		}

		public void deserialize(byte[] bytes) {
			try (DataInputStream is = new DataInputStream(new ByteArrayInputStream(bytes))) {
				for (int i = 0; i < values.length; i++) {
					values[i] = is.readDouble();
				}
			} catch (IOException ignored) {
			}
		}		
	}

	public class DataSession {
		private final Stage[] stages;
		private DataOutputStream output;

		public DataSession(DataOutputStream os) {
			this.output = os;
			this.stages = new Stage[sensorModel.size()];
		}

		public DataSession set(Instant instant) throws IOException {
			if (instant.isBefore(next())) return this;
			if (hasStage()) push();
			saveTimeModel(instant, timeModel.period);
			return this;
		}

		public DataSession set(String measurement, double value) {
			if (!sensorModel.has(measurement)) throw new IllegalArgumentException(measurement + " measurement is unknown");
			return set(sensorModel.get(measurement), value);
		}

		public DataSession set(Magnitude magnitude, double value) {
			if (!sensorModel.has(magnitude)) throw new IllegalArgumentException(magnitude + " measurement is unknown");
			stageOf(magnitude).set(value);
			return this;
		}

		private void saveTimeModel(Instant instant, Period period) throws IOException {
			if (instant.isBefore(next())) return;
			output.close();
			timeModel(period.crop(instant), period);
			output = outputStream();
		}

		public ChronosFile close() throws IOException {
			if (hasStage()) push();
			output.close();
			header.commit();
			return ChronosFile.this;
		}

		private Stage stageOf(Magnitude magnitude) {
			Stage stage = stages[sensorModel.indexOf(magnitude)];
			if (stage == null) stage = stages[sensorModel.indexOf(magnitude)] = new Stage();
			return stage;
		}

		private void push() throws IOException {
			write(new Data(last(), values()));
			header.count++;
			header.step();
			Arrays.fill(stages, null);
		}

		private double[] values() {
			double[] result = new double[sensorModel.size()];
			for (int i = 0; i < result.length; i++)
				result[i] = stages[i] == null ? NaN : measurement(i).reduce(values(i));
			return result;
		}

		private double[] values(int i) {
			return stages[i].values();
		}

		private Magnitude measurement(int i) {
			return sensorModel.magnitudes[i];
		}

		private void write(Data data) throws IOException {
			byte[] bytes = header.compressed ? Compressor.compress(data.serialize()) : data.serialize();
			output.writeShort(Data.MARK);
			output.writeInt(bytes.length);
			output.write(bytes);
		}

		private boolean hasStage() {
			return stream(stages).anyMatch(Objects::nonNull);
		}

	}

	public class Reader {
		private final Consumer<Block> consumer;
		private final Map<Short, BlockReader> readers;
		private TimeModel timeModel;
		private SensorModel sensorModel;
		private Instant instant;

		private Reader(Consumer<Block> consumer) {
			this.consumer = consumer;
			this.readers = readers();
			this.timeModel = null;
			this.sensorModel = null;
			this.instant = null;
		}

		private void read() throws IOException {
			try (DataInputStream is = inputStream()) {
				is.skipBytes(Header.SIZE);
				while (read(is))
					if (is.available() == 0) break;
			}
		}

		private boolean read(DataInputStream is) throws IOException {
			short mark = is.readShort();
			if (!readers.containsKey(mark)) throw new RuntimeException("Corrupted file");
			readers.get(mark).process(is);
			return mark != -1;
		}

		private Map<Short, BlockReader> readers() {
			Map<Short, BlockReader> result = new HashMap<>();
			result.put(Block.NONE, this::skip);
			result.put(TimeModel.MARK, this::readTimeModel);
			result.put(SensorModel.MARK, this::readSensorModel);
			result.put(Data.MARK, this::readData);
			return result;
		}

		private void skip(DataInputStream is) {

		}

		private void readData(DataInputStream is) throws IOException {
			Data data = new Data(instant, new double[sensorModel.size()]);
			data.deserialize(readData(is, is.readInt()));
			consumer.accept(data);
			instant = next(instant);
		}

		private byte[] readData(DataInputStream is, int size) throws IOException {
			byte[] bytes = new byte[size];
			int read = is.read(bytes);
			assert read == size;
			return header.compressed ? Compressor.decompress(bytes) : bytes;
		}

		private void readSensorModel(DataInputStream is) throws IOException {
			sensorModel = new SensorModel(readBigString(is).split("\n"));
			consumer.accept(sensorModel);
		}

		private void readTimeModel(DataInputStream is) throws IOException {
			timeModel = new TimeModel(is.readLong(), is.readShort(), is.readShort());
			consumer.accept(timeModel);
			instant = timeModel.instant;
		}
	}

	public class Query extends Timeline.Builder implements Consumer<Block> {

		public Query() {
			super(count());
		}

		public void accept(Block block) {
			if (block instanceof Data) accept((Data) block);
		}

		private void accept(Data data) {
			set(data.instant);
			for (Magnitude magnitude : sensorModel)
				set(magnitude, data.get(getIndexOf(magnitude)));
		}

		private int getIndexOf(Magnitude magnitude) {
			return sensorModel.indexOf(magnitude);
		}

	}

	public class ItlExporter implements Consumer<Block>, Closeable {
		private final BufferedWriter output;

		public ItlExporter(File file) throws IOException {
			this.output = new BufferedWriter(new FileWriter(file));
			this.output.write("@id " + id() + "\n");
		}

		@Override
		public void accept(Block block) {
			try {
				if (block instanceof TimeModel) accept((TimeModel) block);
				if (block instanceof SensorModel) accept((SensorModel) block);
				if (block instanceof Data) accept((Data) block);
			} catch (IOException e) {
				throw new RuntimeException(e);
			}
		}

		private void accept(SensorModel data) throws IOException {
			output.write("@measurements " + String.join(",",data.magnitudes) + "\n");
		}

		private Period lastPeriod = null;
		private void accept(TimeModel timeModel) throws IOException {
			output.write("@instant " + timeModel.instant.toString() + "\n");
			if (timeModel.period.equals(lastPeriod)) return;
			output.write("@period " + timeModel.period + "\n");
			lastPeriod = timeModel.period;
		}

		private void accept(Data data) throws IOException {
			output.write(toString(data.values) + "\n");
		}

		private String toString(double[] values) {
			return stream(values).mapToObj(this::toString).collect(joining("\t")).stripTrailing();
		}

		private String toString(double value) {
			return isNaN(value) ? "" : format(value);
		}


		private String format(double value) {
			return isNaN(value) ? "" : df.format(value);
		}

		private final DecimalFormat df = formatter();

		private DecimalFormat formatter() {
			DecimalFormat df = new DecimalFormat("#.#############");
			df.setDecimalFormatSymbols(new DecimalFormatSymbols(Locale.ENGLISH));
			return df;
		}

		@Override
		public void close() throws IOException {
			output.close();
		}
	}

	private static class Compressor {

		private static final byte[] buffer = new byte[32768];
		public static byte[] compress(byte[] data) {
			return Zstd.compress(data);
		}

		public static byte[] decompress(byte[] data) {
			long length = Zstd.decompress(buffer,data);
			return Arrays.copyOf(buffer, (int) length);
		}
	}

	private static void writeBigString(DataOutputStream output, String str) throws IOException {
		byte[] bytes = str.getBytes();
		output.writeInt(bytes.length);
		output.write(bytes);
	}

	private String readBigString(DataInputStream is) throws IOException {
		int length = is.readInt();
		byte[] bytes = new byte[length];
		int read = is.read(bytes);
		assert read==length;
		return new String(bytes);
	}

	private static String readBigString(RandomAccessFile raf) throws IOException {
		int length = raf.readInt();
		byte[] bytes = new byte[length];
		int read = raf.read(bytes);
		assert read==length;
		return new String(bytes);
	}

	static class Stage {
		private double[] values;
		private int size;
		private int count = 0;

		public Stage() {
			this.size = 10;
			this.values = new double[size];
		}

		public void set(double value) {
			values[count++] = value;
			if (count >= size) values = resize();
		}

		private double[] resize() {
			this.size += 10;
			double[] result = new double[size];
			arraycopy(values, 0, result, 0, values.length);
			return result;
		}

		public double[] values() {
			return copyOfRange(values, 0, count);
		}

		@Override
		public String toString() {
			return Arrays.toString(values);
		}

	}

	public class Fix implements Closeable {
		private final RandomAccessFile raf;
		private final Instant instant;
		private final long position;

		public Fix(Instant instant) throws IOException {
			this.raf = new RandomAccessFile(file, "rw");
			this.instant = instant;
			this.position = calculatePositionWith(steps());
		}

		private long calculatePositionWith(long steps) {
			if (steps < 0) throw new IllegalArgumentException("Fixing measurement: " + instant + " doesn't exist");
			return checkSealing(file.length() - steps * (Data.SIZE + (long) sensorModel.size() * Double.BYTES));
		}

		private long checkSealing(long result) {
			if (isSealed(result)) throw new IllegalArgumentException("Fixing measurement: " + instant + " is already sealed");
			return result;
		}

		private boolean isSealed(long result) {
			return result < header.timeModelPosition || result < header.sensorModelPosition;
		}

		private long steps() {
			return (header.last.getEpochSecond() - instant.getEpochSecond()) / timeModel().duration();
		}

		public Fix set(String measurement, double value) throws IOException {
			long index = sensorModel.indexOf(new Magnitude(measurement));
			if (index < 0) throw new IllegalArgumentException("Fixing measurement: " + measurement + " doesn't exist");
			raf.seek(position + index * Double.BYTES + Data.SIZE);
			raf.writeDouble(value);
			return this;
		}

		@Override
		public void close() throws IOException {
			raf.close();
		}
	}
}
